import {
  BeforeEventName,
  CommandConfigExtended,
  CommandDefinitionFun,
  CommandLocaleDef,
  CommandOptionConfigWithDescription,
  EventName,
  KoishiCommandDefinition,
  KoishiCommandPutDef,
  KoishiOnContextScope,
  OnContextFunction,
  TopLevelActionDef,
} from '../def';
import 'reflect-metadata';
import {
  Argv,
  Command,
  Dict,
  FieldCollector,
  Selection,
  Session,
  I18n,
} from 'koishi';
import { Metadata } from '../meta/metadata.decorators';
import { CommandPut, DoRegister } from '../registry';
import {
  adaptLocaleDict,
  applyOptionToCommand,
  registerTemplate,
} from '../utility';

// Register method

export const UseMiddleware = (prepend = false): MethodDecorator =>
  DoRegister.middleware(prepend);
export const UseEvent = (name: EventName, prepend = false): MethodDecorator =>
  DoRegister.event({ name, prepend });
export const UseBeforeEvent = (
  name: BeforeEventName,
  prepend = false,
): MethodDecorator => DoRegister.beforeEvent({ name, prepend });
export const UsePlugin = (): MethodDecorator => DoRegister.plugin();

export function UseCommand<D extends string>(
  def: D,
  config?: CommandConfigExtended,
): MethodDecorator;
export function UseCommand<D extends string>(
  def: D,
  desc: string,
  config?: CommandConfigExtended,
): MethodDecorator;
export function UseCommand(
  def: string,
  ...args: [CommandConfigExtended?] | [string, CommandConfigExtended?]
): MethodDecorator {
  const desc = typeof args[0] === 'string' ? (args.shift() as string) : '';
  const config = args[0] as CommandConfigExtended;
  return (obj, key: string, des) => {
    const putOptions: CommandPut.Config<keyof CommandPut.ConfigMap>[] =
      Reflect.getMetadata(KoishiCommandPutDef, obj.constructor, key) ||
      undefined;
    // eslint-disable-next-line @typescript-eslint/ban-types
    const paramTypes: Function[] = Reflect.getMetadata(
      'design:paramtypes',
      obj,
      key,
    );
    const metadataDec = DoRegister.command({
      def,
      desc,
      config,
      putOptions,
      paramTypes,
    });
    return metadataDec(obj, key, des);
  };
}

export const UseFormatter = (name: string) => DoRegister.formatter(name);
export const UserRenderer = (name: string) => DoRegister.renderer(name);

// Context scopes

export const OnContext = (
  ctxFun: OnContextFunction,
): MethodDecorator & ClassDecorator =>
  Metadata.append(KoishiOnContextScope, ctxFun);

export const OnAnywhere = () => OnContext((ctx) => ctx.any());

export const OnNowhere = () => OnContext((ctx) => ctx.never());

export const OnUser = (...values: string[]) =>
  OnContext((ctx) => ctx.user(...values));

export const OnSelf = (...values: string[]) =>
  OnContext((ctx) => ctx.self(...values));

export const OnGuild = (...values: string[]) =>
  OnContext((ctx) => ctx.guild(...values));

export const OnChannel = (...values: string[]) =>
  OnContext((ctx) => ctx.channel(...values));

export const OnPlatform = (...values: string[]) =>
  OnContext((ctx) => ctx.platform(...values));

export const OnPrivate = (...values: string[]) =>
  OnContext((ctx) => ctx.private(...values));

export const OnSelection = (selection: Selection) =>
  OnContext((ctx) => ctx.select(selection));

// Command definition

export const CommandDef = (
  def: CommandDefinitionFun,
): MethodDecorator & ClassDecorator =>
  Metadata.append(KoishiCommandDefinition, def);

export const CommandUse = <T extends Command, R extends any[]>(
  callback: (command: Command, ...args: R) => T,
  ...args: R
) => CommandDef((cmd) => callback(cmd, ...args));

export const CommandLocale = (locale: string, def: CommandLocaleDef) =>
  CommandDef((cmd, ctx) => {
    ctx.i18n.define(locale, `commands.${cmd.name}`, def);
    return cmd;
  });

export const CommandDescription = (desc: string | Dict<string>) => {
  return CommandDef((cmd, ctx) => {
    for (const localData of Object.entries(adaptLocaleDict(desc))) {
      const [locale, text] = localData;
      ctx.i18n.define(locale, `commands.${cmd.name}.description`, text);
    }
    return cmd;
  });
};

export const CommandAlias = (...names: string[]) =>
  CommandDef((cmd) => cmd.alias(...names));

export const CommandShortcut = (
  name: string | RegExp,
  config: Command.Shortcut = {},
) => CommandDef((cmd) => cmd.shortcut(name, config));

export const CommandUsage = (text: Command.Usage) =>
  CommandDef((cmd) => cmd.usage(text));

export const CommandExample = (text: string) =>
  CommandDef((cmd) => cmd.example(text));

export const CommandOption = (
  name: string,
  desc: string,
  config: CommandOptionConfigWithDescription = {},
) =>
  CommandDef((cmd, ctx) =>
    applyOptionToCommand(ctx, cmd, { name, desc, config }),
  );

export const CommandUserFields = (fields: FieldCollector<'user'>) =>
  CommandDef((cmd) => cmd.userFields(fields));

export const CommandChannelFields = (fields: FieldCollector<'channel'>) =>
  CommandDef((cmd) => cmd.channelFields(fields));

export const CommandBefore = (callback: Command.Action, append = false) =>
  CommandDef((cmd) => cmd.before(callback, append));

export const CommandAction = (callback: Command.Action, prepend = false) =>
  CommandDef((cmd) => cmd.action(callback, prepend));

export const CommandTemplate = (name: string, text: string | Dict<string>) =>
  CommandDef((cmd, ctx) => {
    registerTemplate({ name, text: adaptLocaleDict(text) }, ctx, cmd);
    return cmd;
  });

// Command put config

export const PutArgv = (field?: keyof Argv) =>
  field ? CommandPut.decorate('argvField', field) : CommandPut.decorate('argv');
export const PutSession = (field?: keyof Session) =>
  field ? CommandPut.decorate('sessionField', field) : PutArgv('session');
export const PutArg = (index: number, decl?: Argv.Declaration) =>
  CommandPut.decorate('arg', { index, decl });
export const PutArgs = () => CommandPut.decorate('args');
export const PutOption = (
  name: string,
  desc: string,
  config: CommandOptionConfigWithDescription = {},
) => CommandPut.decorate('option', { name, desc, config });

export const PutUser = (field: FieldCollector<'user'>) =>
  CommandPut.decorate('user', field);

export const PutChannel = (field: FieldCollector<'channel'>) =>
  CommandPut.decorate('channel', field);

export const PutGuild = (field: FieldCollector<'channel'>) =>
  CommandPut.decorate('guild', field);

export const PutUserName = (useDatabase = true) =>
  CommandPut.decorate('username', useDatabase);

export const PutUserId = () => PutSession('userId');
export const PutGuildId = () => PutSession('guildId');
export const PutGuildName = () => PutSession('guildName');
export const PutChannelId = () => PutSession('channelId');
export const PutChannelName = () => PutSession('channelName');
export const PutSelfId = () => PutSession('selfId');
export const PutBot = () => PutSession('bot');
export const PutNext = () => PutArgv('next');
export const PutRenderer = (path: string) =>
  CommandPut.decorate('renderer', path);
export const PutCommonRenderer = () => CommandPut.decorate('renderer');
export const PutTemplate = (name: string, text: string | Dict<string>) =>
  CommandPut.decorate('template', {
    name,
    text: adaptLocaleDict(text),
  });
export const PutObject = () => CommandPut.decorate('typeClass');

export const TopLevelAction = (action: TopLevelActionDef): ClassDecorator =>
  Metadata.append('KoishiTopLevelAction', action);

export const DefineTemplate = (name: string, text: string | Dict<string>) =>
  TopLevelAction((ctx) =>
    registerTemplate({ name, text: adaptLocaleDict(text) }, ctx),
  );

export function DefineLocale(locale: string, dict: I18n.Store): ClassDecorator;
export function DefineLocale(
  locale: string,
  key: string,
  value: I18n.Node,
): ClassDecorator;
export function DefineLocale(
  locale: string,
  ...args: [I18n.Store] | [string, I18n.Node]
): ClassDecorator {
  return TopLevelAction((ctx) =>
    // eslint-disable-next-line @typescript-eslint/ban-ts-comment
    // @ts-ignore
    ctx.i18n.define(locale, ...args),
  );
}
