// import 'source-map-support/register';
import { DrawPluginConfig } from './config';
import {
  DefinePlugin,
  InjectLogger,
  LifecycleEvents,
  UseCommand,
  CommandLocale,
  PutArg,
  PutCommonRenderer,
  CRenderer,
  PutUserName,
  Renderer,
  PutRenderer,
  StarterPlugin,
} from 'koishi-thirdeye';
import { Logger } from 'koishi';
import path from 'path';
import loadJsonFile from 'load-json-file';
import _ from 'lodash';
import * as localeZh from './locales/zh';
import * as localeEn from './locales/en';
import leven from 'leven';
import { Decks, Drawer } from './drawer';
import { YamlDrawer, YamlStruct } from './yaml-plugin';
export * from './config';
import yaml from 'js-yaml';
import fs from 'fs';

@DefinePlugin()
export default class DrawPlugin
  extends StarterPlugin(DrawPluginConfig)
  implements LifecycleEvents
{
  @InjectLogger()
  private logger: Logger;

  jsonDeckFiles = new Map<string, string[]>();
  drawer: Drawer;
  yamlDecks = new Map<string, YamlDrawer>();
  yamlDeckFiles = new Map<string, YamlDrawer>();

  onApply() {
    this.drawer = new Drawer({}, this.logger, this.config.maxDepth);
  }

  async loadJsonDeck(file: string) {
    const filename = path.basename(file);
    if (this.jsonDeckFiles.has(filename)) return;
    try {
      const content: Decks = await loadJsonFile(file);
      const deckTitles = Object.keys(content);
      this.jsonDeckFiles.set(filename, deckTitles);
      for (const key of deckTitles) {
        if (this.drawer.decks[key]) {
          this.logger.warn(`Duplicate deck ${key} in ${file}`);
        }
        this.drawer.decks[key] = content[key];
      }
    } catch (e) {
      this.logger.error(`Load deck file ${file} failed: ${e.message}`);
    }
  }

  async loadYamlDeck(file: string) {
    const filename = path.basename(file);
    if (this.yamlDeckFiles.has(filename)) return;
    try {
      const content = yaml.load(
        await fs.promises.readFile(file, 'utf8'),
      ) as YamlStruct;
      if (!content.includes || !content.command) {
        throw new Error('Invalid yaml deck');
      }
      const deck = new YamlDrawer(content, this.logger, this.config.maxDepth);
      this.yamlDeckFiles.set(filename, deck);
      this.yamlDecks.set(deck.meta.command, deck);
    } catch (e) {
      this.logger.error(`Load deck file ${file} failed: ${e.message}`);
    }
  }

  async loadJsonDecks() {
    this.jsonDeckFiles.clear();
    this.drawer.decks = {};
    const files = await this.config.loadFileList('json');
    await Promise.all(files.map((file) => this.loadJsonDeck(file)));
    const deckCount = _.sumBy(
      Array.from(this.jsonDeckFiles.values()),
      (v) => v.length,
    );
    const deckFileCount = this.jsonDeckFiles.size;
    this.logger.info(
      `Loaded ${deckCount} JSON decks from ${deckFileCount} deck files.`,
    );
    return { deckCount, deckFileCount };
  }

  async loadYamlDecks() {
    this.yamlDeckFiles.clear();
    this.yamlDecks.clear();
    const files = await this.config.loadFileList('yaml', 'yml');
    await Promise.all(files.map((file) => this.loadYamlDeck(file)));
    const deckCount = this.yamlDecks.size;
    const deckFileCount = deckCount;
    this.logger.info(
      `Loaded ${deckCount} YAML decks from ${deckFileCount} deck files.`,
    );
    return { deckCount, deckFileCount };
  }

  async loadDecks() {
    const results = await Promise.all([
      this.loadJsonDecks(),
      this.loadYamlDecks(),
    ]);
    return {
      deckCount: results[0].deckCount + results[1].deckCount,
      deckFileCount: results[0].deckFileCount + results[1].deckFileCount,
    };
  }

  async onConnect() {
    await this.loadDecks();
  }

  @UseCommand('draw <name> [param]', { checkArgCount: true })
  @CommandLocale('zh', localeZh.commands.draw)
  @CommandLocale('en', localeEn.commands.draw)
  drawCommand(
    @PutArg(0) name: string,
    @PutArg(1) param: string,
    @PutUserName() user: string,
    @PutCommonRenderer() renderer: CRenderer,
  ) {
    if (this.yamlDecks.has(name)) {
      const deck = this.yamlDecks.get(name);
      const result = deck.draw(param, user);
      return result ?? renderer('.notFound');
    }
    const result = this.drawer.drawFromDeck(name);
    if (!result) {
      return renderer('.notFound');
    }
    return (
      renderer('.result', {
        user,
      }) +
      '\n' +
      result
    );
  }

  @UseCommand('draw.list')
  @CommandLocale('zh', localeZh.commands.list)
  @CommandLocale('en', localeEn.commands.list)
  onListCommand(
    @PutRenderer('.fileInfo') renderer: Renderer,
    @PutRenderer('.yamlFileInfo') yamlRenderer: Renderer,
  ) {
    const jsonEntries = Array.from(this.jsonDeckFiles.entries()).map(
      ([file, titles]) => ({
        file,
        count: titles.length,
      }),
    );
    const yamlEntries = Array.from(this.yamlDeckFiles.entries()).map(
      ([file, deck]) => ({ ...deck.meta, file }),
    );
    return [
      ...jsonEntries.map((entry) => renderer(entry)),
      ...yamlEntries.map((entry) => yamlRenderer(entry)),
    ].join('\n');
  }

  @UseCommand('draw.help')
  @CommandLocale('zh', localeZh.commands.help)
  @CommandLocale('en', localeEn.commands.help)
  onHelpCommand(
    @PutRenderer('.fileInfo') renderer: Renderer,
    @PutRenderer('.yamlFileInfo') yamlRenderer: Renderer,
  ) {
    const jsonEntries = Array.from(this.jsonDeckFiles.entries()).map(
      ([file, titles]) => ({
        file,
        count: titles.length,
        titles,
      }),
    );
    const yamlDecks = Array.from(this.yamlDeckFiles.entries()).map(
      ([file, deck]) => ({ deck, file }),
    );
    return [
      ...jsonEntries.map(
        (entry) =>
          renderer(entry) +
          '\n' +
          entry.titles.map((title) => `draw ${title}`).join('\n'),
      ),
      ...yamlDecks.map(
        (entry) =>
          yamlRenderer(entry.deck.meta) +
          '\n' +
          entry.deck
            .getCommands()
            .map((command) => `draw ${command}`)
            .join('\n'),
      ),
    ].join('\n');
  }

  @UseCommand('draw.reload')
  @CommandLocale('zh', localeZh.commands.reload)
  @CommandLocale('en', localeEn.commands.reload)
  async onReloadCommand(@PutRenderer('.result') renderer: Renderer) {
    const result = await this.loadDecks();
    return renderer(result);
  }

  @UseCommand('draw.search <word:text>', { checkArgCount: true })
  @CommandLocale('zh', localeZh.commands.search)
  @CommandLocale('en', localeEn.commands.search)
  async onSearchCommand(
    @PutArg(0) word: string,
    @PutRenderer('.result') renderer: Renderer,
    @PutRenderer('.notFound') notFoundRenderer: Renderer,
  ) {
    const allDecks = _.flatten([
      ...Array.from(this.jsonDeckFiles.values()),
      ...Array.from(this.yamlDeckFiles.values()).map((v) => v.getCommands()),
    ]).filter((d) => d.includes(word));
    if (!allDecks.length) {
      return notFoundRenderer({ word });
    }
    _.sortBy(allDecks, (deck) => leven(deck, word));
    return `${renderer()}\n${allDecks.join('|')}`;
  }
}
