import { Logger, Random } from 'koishi';
import _ from 'lodash';
import { OneDice } from 'onedice';

export type Decks = Record<string, string[]>;

export class Drawer {
  constructor(
    public decks: Decks,
    private logger: Logger,
    private maxDepth: number,
  ) {}

  protected parseEntry(name: string, entry: string, depth = 1): string {
    let result = entry.replace(/\[([^\]]+)\]/g, (dicePattern) =>
      new OneDice().calculate(dicePattern.slice(1, -1)).toString(),
    );
    if (depth > this.maxDepth) {
      this.logger.warn(`Max depth ${this.maxDepth} reached in deck ${name}.`);
      return entry;
    }
    const usedUniqueIndex = new Map<string, Set<number>>();
    result = result
      .replace(/\{[^%\}]+\}/g, (refPattern) => {
        let deckName = refPattern.slice(1, -1);
        if (deckName.startsWith('$')) {
          deckName = deckName.slice(1);
        }
        return this.drawFromDeck(deckName, depth + 1, name) || refPattern;
      })
      .replace(/\{%[^%\}]+\}/g, (uniqRefPattern) => {
        const refName = uniqRefPattern.slice(2, -1);
        let indexArray = usedUniqueIndex.get(refName);
        if (!indexArray) {
          indexArray = new Set();
          usedUniqueIndex.set(refName, indexArray);
        }
        const deck = this.decks[refName];
        if (!deck) {
          this.logger.warn(
            `Referenced deck ${refName} not found in deck ${name}.`,
          );
          return uniqRefPattern;
        }

        const availableIndexes = _.range(deck.length).filter(
          (index) => !indexArray.has(index),
        );
        if (availableIndexes.length === 0) {
          this.logger.warn(
            `No more unique entries left for ${refName} in deck ${name}.`,
          );
          return uniqRefPattern;
        }
        const index = Random.pick(availableIndexes);
        indexArray.add(index);

        const entry = deck[index];
        return this.parseEntry(refName, entry, depth + 1);
      });
    return result;
  }

  drawFromDeck(name: string, depth = 1, referencedDeck?: string): string {
    const deck = this.decks[name];
    if (!deck) {
      this.logger.warn(
        `${referencedDeck ? 'Referenced deck' : 'Deck'} ${name} not found${
          referencedDeck ? `in deck ${referencedDeck}` : ''
        }.`,
      );
      return null;
    }
    const entry = Random.pick(deck);
    return this.parseEntry(name, entry, depth);
  }
}
