mod router;
use crate::router::{Router, SECRET_LENGTH};
use crossbeam_utils::thread;
use std::env;
use std::error::Error;
use std::io::{Read, Write};
use std::mem::MaybeUninit;
use std::mem::{size_of, transmute};
use std::sync::atomic::Ordering;

#[repr(C)]
pub struct Meta {
    pub src_id: u8,
    pub dst_id: u8,
    pub reversed: u16,
}

use serde::Deserialize;

#[derive(Deserialize)]
pub struct Config {
    pub local_id: u8,
    pub local_secret: String,
    pub routers: Vec<ConfigRouter>,
}
#[derive(Deserialize)]
pub struct ConfigRouter {
    pub remote_id: u8,
    pub proto: u8,
    pub family: u8,
    pub mark: u32,
    pub endpoint: String,
    pub remote_secret: String,
    pub dev: String,
    pub up: String,
}

fn main() -> Result<(), Box<dyn Error>> {
    println!("Starting");
    let config: Config = serde_json::from_str(env::args().nth(1).ok_or("need param")?.as_str())?;
    let local_secret: [u8; SECRET_LENGTH] = Router::create_secret(config.local_secret.as_str())?;
    let routers: Vec<Router> = config
        .routers
        .into_iter()
        .map(|c| Router::new(c, config.local_id))
        .collect::<Result<Vec<_>, _>>()?;

    println!("created tuns");
    const META_SIZE: usize = size_of::<Meta>();

    thread::scope(|s| {
        for router in routers {
            let (mut reader, mut writer) = router.split();

            s.spawn(move |_| {
                let mut buffer = [0u8; 1500 - 20]; // minus typical IP header space

                // Pre-initialize with our Meta header (local -> remote)
                let meta = Meta {
                    src_id: config.local_id,
                    dst_id: reader.config.remote_id,
                    reversed: 0,
                };
                // Turn the Meta struct into bytes
                let meta_bytes: &[u8; META_SIZE] =
                    unsafe { &*(&meta as *const Meta as *const [u8; META_SIZE]) };
                buffer[..META_SIZE].copy_from_slice(meta_bytes);

                loop {
                    let n = reader.tun_reader.read(&mut buffer[META_SIZE..]).unwrap();
                    let guard = crossbeam::epoch::pin();
                    let shared = reader.endpoint.load(Ordering::Acquire, &guard);
                    if let Some(addr) = unsafe { shared.as_ref() } {
                        reader.encrypt(&mut buffer[META_SIZE..META_SIZE + n]);
                        let _ = reader.socket.send_to(&buffer[..META_SIZE + n], addr);
                    }
                }
            });

            s.spawn(move |_| {
                let mut recv_buf = [MaybeUninit::uninit(); 1500];
                loop {
                    let _ = (|| {
                        let (len, addr) = writer.socket.recv_from(&mut recv_buf).unwrap();
                        let packet: &mut [u8] = unsafe { transmute(&mut recv_buf[..len]) };
                        // if addr.is_ipv6() { println!("{:X?}", packet) }
                        let offset = if addr.is_ipv4() {
                            (packet[0] & 0x0f) as usize * 4
                        } else {
                            0
                        } + META_SIZE;

                        let guard = crossbeam::epoch::pin();
                        let current_shared = writer.endpoint.load(Ordering::SeqCst, &guard);
                        let is_same = unsafe { current_shared.as_ref() }
                            .map(|c| *c == addr)
                            .unwrap_or(false);
                        if !is_same {
                            let new_shared = crossbeam::epoch::Owned::new(addr).into_shared(&guard);
                            let old_shared =
                                writer.endpoint.swap(new_shared, Ordering::SeqCst, &guard);
                            unsafe {
                                guard.defer_destroy(old_shared);
                            }
                        }

                        let payload = &mut packet[offset..];
                        writer.decrypt(payload, &local_secret);
                        writer.tun_writer.write_all(payload)?;

                        Ok::<(), Box<dyn Error>>(())
                    })();
                }
            });
        }
    })
    .unwrap();
    Ok(())
}
