use crate::config::{ConfigRouter, Schema};
use anyhow::{bail, ensure, Result};
use crossbeam::epoch::{pin, Atomic};
use libc::{
    setsockopt, sock_filter, sock_fprog, socklen_t, BPF_ABS, BPF_B, BPF_H, BPF_IND, BPF_JEQ, BPF_JMP, BPF_K, BPF_LD, BPF_LDX, BPF_MSH, BPF_RET,
    MSG_FASTOPEN, SOL_SOCKET, SO_ATTACH_REUSEPORT_CBPF,
};
use socket2::{Domain, Protocol, SockAddr, SockFilter, Socket, Type};
use std::fs::{File, OpenOptions};
use std::net::Shutdown;
use std::{
    ffi::c_void,
    mem::MaybeUninit,
    net::{Ipv4Addr, Ipv6Addr, SocketAddrV4, SocketAddrV6, ToSocketAddrs},
    os::fd::{AsRawFd, FromRawFd},
    process::{Command, ExitStatus},
    sync::atomic::Ordering,
};

pub const SECRET_LENGTH: usize = 32;
pub const META_SIZE: usize = 4;

pub struct Router {
    pub config: ConfigRouter,
    pub tun: File,
    pub socket: Socket,
    pub endpoint: Atomic<SockAddr>,

    pub tcp_listener_connection: Atomic<Socket>,
}

impl Router {
    pub(crate) fn decrypt(&self, data: &mut [u8]) {
        for (i, b) in data.iter_mut().enumerate() {
            *b ^= self.config.local_secret[i % SECRET_LENGTH];
        }
    }
    pub(crate) fn encrypt(&self, data: &mut [u8]) {
        for (i, b) in data.iter_mut().enumerate() {
            *b ^= self.config.remote_secret[i % SECRET_LENGTH];
        }
    }

    pub fn create_socket(config: &ConfigRouter) -> Result<Socket> {
        println!("create_socket {}", config.remote_id);
        match config.schema {
            Schema::IP => {
                let result = Socket::new(config.family, Type::RAW, Some(Protocol::from(config.proto as i32)))?;
                if config.mark != 0 {
                    result.set_mark(config.mark)?;
                }
                Self::attach_filter_ip(config, &result)?;
                Ok(result)
            }
            Schema::UDP => {
                let result = Socket::new(config.family, Type::DGRAM, Some(Protocol::UDP))?;
                if config.mark != 0 {
                    result.set_mark(config.mark)?;
                }
                if config.src_port != 0 {
                    result.set_reuse_port(true)?;
                    let addr = Self::bind_addr(config);
                    result.bind(&addr)?;
                }
                Ok(result)
            }
            Schema::TCP => Ok(unsafe { Socket::from_raw_fd(0) }),
        }
    }

    pub fn listen_tcp(&self) -> Socket {
        // listener
        let result = Socket::new(Domain::IPV6, Type::STREAM, Some(Protocol::TCP)).unwrap();
        result.set_reuse_address(true).unwrap();
        let addr = SockAddr::from(SocketAddrV6::new(Ipv6Addr::UNSPECIFIED, self.config.src_port, 0, 0));
        result.bind(&addr).unwrap();
        result.listen(100).unwrap();
        result
    }

    pub fn connect_tcp(&self) -> Result<Socket> {
        // tcp client 的 socket 不要在初始化时创建，在循环里创建
        // 创建 socket 和 获取 endpoint 失败会 panic，连接失败会 error
        let result = Socket::new(self.config.family, Type::STREAM, Some(Protocol::TCP)).unwrap();
        result.set_tcp_nodelay(true).unwrap();
        if self.config.mark != 0 {
            result.set_mark(self.config.mark).unwrap();
        }
        if self.config.src_port != 0 {
            result.set_reuse_address(true).unwrap();
            let addr = Self::bind_addr(&self.config);
            result.bind(&addr)?;
        }

        let guard = pin();
        let endpoint_ref = self.endpoint.load(Ordering::Relaxed, &guard);
        let endpoint = unsafe { endpoint_ref.as_ref() }.unwrap();

        result.send_to_with_flags(&[self.config.local_id, self.config.remote_id], endpoint, MSG_FASTOPEN)?;
        Ok(result)
    }

    fn attach_filter_ip(config: &ConfigRouter, socket: &Socket) -> Result<()> {
        // 由于多个对端可能会使用相同的 ipprpto 号，这里确保每个 socket 上只会收到自己对应的对端发来的消息
        let value = u16::from_be_bytes([config.remote_id, config.local_id]) as u32;

        // 如果你的协议是 UDP，这里必须是 8 (跳过 UDP 头: SrcPort(2)+DstPort(2)+Len(2)+Sum(2))
        // 如果是纯自定义 IP 协议，这里是 0
        let payload_offset = 0;

        // IP filter 工作原理：
        // 每个对端起一个 raw socket
        // 根据报文内容判断是给谁的。拒绝掉不是给自己的报文
        // IPv4 raw socket 带 IP 头，IPv6 不带

        let filters: &[SockFilter] = match socket.domain()? {
            Domain::IPV4 => &[
                // [IPv4] 计算 IPv4 头长度: X = 4 * (IP[0] & 0xf)
                bpf_stmt(BPF_LDX | BPF_B | BPF_MSH, 0),
                // A = Packet[X + payload_offset]
                bpf_stmt(BPF_LD | BPF_H | BPF_IND, payload_offset),
                // if (A == target_val) goto Accept; else goto Reject;
                bpf_jump(BPF_JMP | BPF_JEQ | BPF_K, value, 0, 1),
                // 【接受 (True 路径)】
                bpf_stmt(BPF_RET | BPF_K, u32::MAX),
                // 【拒绝 (False 路径)】
                bpf_stmt(BPF_RET | BPF_K, 0),
            ],
            Domain::IPV6 => &[
                // raw socket IPv6 没有 header，加载第 0 字节到累加器 A
                bpf_stmt(BPF_LD | BPF_H | BPF_ABS, 0),
                // if (A == target_val) goto Accept; else goto Reject;
                bpf_jump(BPF_JMP | BPF_JEQ | BPF_K, value, 0, 1),
                // 【接受 (True 路径)】
                bpf_stmt(BPF_RET | BPF_K, u32::MAX),
                // 【拒绝 (False 路径)】
                bpf_stmt(BPF_RET | BPF_K, 0),
            ],
            _ => bail!("unsupported family"),
        };
        socket.attach_filter(filters)?;

        Ok(())
    }

    pub fn attach_filter_udp(group: Vec<&Router>) -> Result<()> {
        let values: Vec<u16> = group.iter().map(|&f| u16::from_be_bytes([f.config.remote_id, f.config.local_id])).collect();

        let mut filters: Vec<SockFilter> = Vec::with_capacity(1 + values.len() * 2 + 1);

        // udp filter 工作原理：
        // 每个对端起一个 udp socket
        // 根据报文内容判断是给谁的，调度给对应的端口复用组序号

        // Load the first 4 bytes of the packet into the accumulator (A)
        filters.push(bpf_stmt(BPF_LD | BPF_H | BPF_ABS, 0));
        for (i, &val) in values.iter().enumerate() {
            // 如果匹配，继续下一句(返回)，如果不匹配，跳过下一句。
            filters.push(bpf_jump(BPF_JMP | BPF_JEQ | BPF_K, val as u32, 0, 1));
            // If match, return the index (i + 1, since 0 means drop)
            filters.push(bpf_stmt(BPF_RET | BPF_K, i as u32));
        }
        // If no match found after all comparisons, drop the packet
        filters.push(bpf_stmt(BPF_RET | BPF_K, u32::MAX));

        let prog = sock_fprog {
            len: filters.len() as u16,
            filter: filters.as_mut_ptr() as *mut sock_filter,
        };
        let fd = group[0].socket.as_raw_fd();
        let ret = unsafe {
            setsockopt(
                fd,
                SOL_SOCKET,
                SO_ATTACH_REUSEPORT_CBPF,
                &prog as *const _ as *const c_void,
                size_of_val(&prog) as socklen_t,
            )
        };
        ensure!(ret != -1, std::io::Error::last_os_error());
        Ok(())
    }

    pub(crate) fn handle_outbound_ip_udp(&self) {
        let mut buffer = [0u8; 1504];

        loop {
            let n = self.recv_tun(&mut buffer[..]).unwrap();
            let payload = &mut buffer[..n];
            payload[0] = self.config.local_id;
            payload[1] = self.config.remote_id;
            let guard = pin();
            let endpoint_ref = self.endpoint.load(Ordering::Relaxed, &guard);
            if let Some(endpoint) = unsafe { endpoint_ref.as_ref() } {
                self.encrypt(&mut payload[META_SIZE..]);
                let _ = self.socket.send_to(&payload, endpoint);
            }
        }
    }

    pub(crate) fn handle_inbound_ip_udp(&self) {
        let mut buffer = [MaybeUninit::uninit(); 1500];
        loop {
            // 收到一个非法报文只丢弃一个报文
            let (len, addr) = { self.socket.recv_from(&mut buffer).unwrap() }; // recv 出错直接 panic
            {
                let guard = pin();
                let current_shared = self.endpoint.load(Ordering::Relaxed, &guard);
                let is_same = unsafe { current_shared.as_ref() }.map(|c| *c == addr).unwrap_or(false);
                if !is_same {
                    let new_shared = crossbeam::epoch::Owned::new(addr).into_shared(&guard);
                    let old_shared = self.endpoint.swap(new_shared, Ordering::Release, &guard);
                    unsafe { guard.defer_destroy(old_shared) }
                }
            }

            let packet: &mut [u8] = unsafe { std::mem::transmute(&mut buffer[..len]) };
            // 只有 ipv4 raw 会给 IP报头
            let offset = if self.config.family == Domain::IPV4 && self.config.schema == Schema::IP {
                (packet[0] & 0x0f) as usize * 4
            } else {
                0
            };
            let payload = &mut packet[offset..];
            payload[0] = 0;
            payload[1] = 0;
            self.decrypt(&mut payload[META_SIZE..]);
            let _ = self.send_tun(payload);
        }
    }

    pub(crate) fn handle_outbound_tcp(&self, connection: &Socket) {
        let _ = (|| -> Result<()> {
            let mut buffer = [0u8; 1500];
            loop {
                let n = self.recv_tun(&mut buffer)?;
                let payload = &mut buffer[..n];
                let len_bytes = (payload.len() as u16).to_le_bytes();
                payload[0] = len_bytes[0];
                payload[1] = len_bytes[1];
                self.encrypt(&mut payload[META_SIZE..]);
                Router::send_all_tcp(&connection, &payload)?;
            }
        })();
        let _ = connection.shutdown(Shutdown::Both);
    }
    pub(crate) fn handle_inbound_tcp(&self, connection: &Socket) {
        let _ = (|| -> Result<()> {
            let mut buffer = [0u8; 1500];
            loop {
                Router::recv_exact_tcp(&connection, &mut buffer[..2])?;
                let len = u16::from_le_bytes([buffer[0], buffer[1]]) as usize;
                Router::recv_exact_tcp(&connection, &mut buffer[2..len])?;
                let payload = &mut buffer[..len];
                self.decrypt(&mut payload[META_SIZE..]);
                payload[0] = 0;
                payload[1] = 0;
                self.send_tun(payload)?;
            }
        })();
        let _ = connection.shutdown(Shutdown::Both);
    }

    pub(crate) fn recv_exact_tcp(sock: &Socket, buf: &mut [u8]) -> Result<()> {
        let mut buf: &mut [MaybeUninit<u8>] = unsafe { std::mem::transmute(buf) };
        while !buf.is_empty() {
            let n = sock.recv(buf)?;
            ensure!(n != 0, std::io::ErrorKind::UnexpectedEof);
            buf = &mut buf[n..];
        }
        Ok(())
    }

    pub(crate) fn send_all_tcp(sock: &Socket, mut buf: &[u8]) -> Result<()> {
        while !buf.is_empty() {
            let n = sock.send(buf)?;
            buf = &buf[n..];
        }
        Ok(())
    }

    fn create_tun_device(config: &ConfigRouter) -> File {
        let file = OpenOptions::new().read(true).write(true).open("/dev/net/tun").unwrap();
        let fd = file.as_raw_fd();

        unsafe {
            let mut ifr: libc::ifreq = std::mem::zeroed();
            // 关键：只设 IFF_TUN，不设 IFF_NO_PI
            ifr.ifr_ifru.ifru_flags = libc::IFF_TUN as i16;

            let mut name_bytes = config.dev.as_bytes().to_vec();
            name_bytes.push(0);
            libc::strncpy(ifr.ifr_name.as_mut_ptr(), name_bytes.as_ptr() as *const libc::c_char, 16);
            let res = libc::ioctl(fd, 0x400454ca, &ifr); // TUNSETIFF 的编码
            println!("{}", res);
            if res < 0 {
                panic!("ioctl failed");
            }
        }
        file
    }

    fn recv_tun(&self, buf: &mut [u8]) -> Result<usize> {
        let n = unsafe { libc::read(self.tun.as_raw_fd(), buf.as_mut_ptr() as *mut c_void, buf.len()) };
        ensure!(n > 0, "Read error: {}", std::io::Error::last_os_error());
        Ok(n as usize)
    }

    fn send_tun(&self, buf: &[u8]) -> Result<()> {
        let n = unsafe { libc::write(self.tun.as_raw_fd(), buf.as_ptr() as *mut c_void, buf.len()) };
        ensure!(n > 0, "Write error: {}", std::io::Error::last_os_error());
        Ok(())
    }

    fn run_up_script(config: &ConfigRouter) -> Result<ExitStatus> {
        Ok(Command::new("sh").args(["-c", config.up.as_str()]).status()?)
    }

    fn create_endpoint(config: &ConfigRouter) -> Atomic<SockAddr> {
        println!("create_endpoint {}", config.remote_id);
        match (config.endpoint.clone(), config.dst_port)
            .to_socket_addrs()
            .unwrap_or_default()
            .filter(|a| match config.family {
                Domain::IPV4 => a.is_ipv4(),
                Domain::IPV6 => a.is_ipv6(),
                _ => false,
            })
            .next()
        {
            None => Atomic::null(),
            Some(addr) => Atomic::new(addr.into()),
        }
    }

    pub fn new(config: ConfigRouter) -> Result<Router> {
        println!("creating {}", config.remote_id);
        let router = Router {
            tun: Self::create_tun_device(&config),
            endpoint: Self::create_endpoint(&config),
            socket: Self::create_socket(&config)?,
            tcp_listener_connection: Atomic::null(),
            config,
        };
        println!("run_up_script {}", &router.config.remote_id);
        Self::run_up_script(&router.config)?;
        Ok(router)
    }

    fn bind_addr(config: &ConfigRouter) -> SockAddr {
        match config.family {
            Domain::IPV4 => SockAddr::from(SocketAddrV4::new(Ipv4Addr::UNSPECIFIED, config.src_port)),
            Domain::IPV6 => SockAddr::from(SocketAddrV6::new(Ipv6Addr::UNSPECIFIED, config.src_port, 0, 0)),
            _ => panic!("unsupported family"),
        }
    }
}

fn bpf_stmt(code: u32, k: u32) -> SockFilter {
    SockFilter::new(code as u16, 0, 0, k)
}

fn bpf_jump(code: u32, k: u32, jt: u8, jf: u8) -> SockFilter {
    SockFilter::new(code as u16, jt, jf, k)
}
