import parse from 'csv-parse';
import util from 'util';
import fs from 'fs';
import path from 'path';
import YAML from 'yaml';
import _, { add } from 'lodash';
import child_process from 'child_process';
import assert from 'assert';
import { promises as dns } from 'dns';
import os from 'os';
import { getRules } from 'ip-pcc';
import { Rule } from 'ip-pcc/dist/src';

async function generateOcpasswdLine(username: string, password: string) {
  const tmpName = os.tmpdir() + '/' + Math.floor(Math.random() * 10000000);
  await util.promisify(child_process.exec)(`echo "${password}\\n${password}" | ocpasswd -c ${tmpName} ${username}`);
  const res = (await fs.promises.readFile(tmpName, 'utf-8')).trim();
  await fs.promises.unlink(tmpName);
  return res;
}

interface GatewayGroup extends Record<string, any> {
  id: number;
  name: string;
  locationPrefix: string;
  includeRouters: string;
  excludeRouters: string;
  children: string;
  destMark: number;
}

interface GostRoute {
  Retries?: string;
  ServeNodes: string[];
  ChainNodes?: string[];
  Mark?: number
}

interface GostConfig extends Partial<GostRoute> {
  Routes: GostRoute[];
  Debug?: boolean;
}

type CommonEntry = Record<string, any>;

interface Gateway {
  [key: string]: any;
  pccWeight?: number;
  pccRules?: Rule[];
  noOut: boolean;
  hidden: boolean;
  badUdp: boolean;
  redirectAllPorts: boolean;
  tunProxy: string;
}

class InventoryBuilder {
  hosts: { [key: string]: CommonEntry };
  gateways: Record<string, Record<string, Gateway>>;
  gatewayGroups: GatewayGroup[];
  connections: string[];
  routeLists: Record<string, string[]>;
  resolveCache: Map<string, string>;
  resolver: dns.Resolver;
  vars: CommonEntry;
  linksOnly: string[];
  linksLimit: string[];
  protosOnly: string[];

  constructor() {
    this.resolveCache = new Map();
    this.resolver = new dns.Resolver();
    this.resolver.setServers(process.env.DNS ? [process.env.DNS] : ['114.114.114.114', '223.5.5.5']);
  }

  getDockerImageTag(host: CommonEntry) {
    if (host.arch && host.arch.length) {
      return `:master-${host.arch}`;
    } else {
      return '';
    }
  }

  domainResolveFunctions = new Map<string, ((result: string) => void)[]>();

  async resolveDomainProcess(domain: string, ipv6: boolean) {
    const cacheKey = `${domain}-${ipv6 ? ':v6' : 'v4'}`;
    if (this.resolveCache.has(cacheKey)) {
      return this.resolveCache.get(cacheKey);
    }
    const rrtype = (domain.includes('-v6') || ipv6) ? 'AAAA' : 'A';
    let resolvedIP: string;
    while (true) {
      try {
        [resolvedIP] = (await this.resolver.resolve(domain, rrtype)) as string[];
        break;
      } catch (e) {
        console.log(`${domain} => FAIL: ${e.toString()}`);
      }
    }
    if (rrtype === 'AAAA') {
      resolvedIP = `[${resolvedIP}]`;
    }
    console.log(`${domain} => ${resolvedIP}`);
    this.resolveCache.set(cacheKey, resolvedIP);
    return resolvedIP;
  }

  async resolveDomain(domain: string, ipv6: boolean) {
    if (!domain || domain.match(/(\d{1,3}\.){3}\d{1,3}/)) {
      return domain;
    }
    return new Promise<string>(async(resolve) => {
      const cacheKey = `${domain}-${ipv6 ? ':v6' : ':v4'}`;
      if (this.resolveCache.has(cacheKey)) {
        resolve(this.resolveCache.get(cacheKey));
      } else {
        let resolveFunctions = this.domainResolveFunctions.get(cacheKey);
        if (!resolveFunctions) { 
          resolveFunctions = [];
          this.domainResolveFunctions.set(cacheKey, resolveFunctions);
          const result = await this.resolveDomainProcess(domain, ipv6);
          resolve(result);
          resolveFunctions.forEach(f => f(result));
          this.domainResolveFunctions.delete(cacheKey);
        } else {
          resolveFunctions.push(resolve);
        }
      }
    })
  }

  async load(sheetName: string): Promise<Record<string, any>[]> {
    const data = await fs.promises.readFile(path.join('data', `内网互联计划 - ${sheetName}.csv`));
    // @ts-ignore
    return (await util.promisify(parse)(data, { columns: true, cast: true })).filter(h => h.id);
  }

  async loadGateways() {
    const gateways = await this.load('gateways2');
    for (let gateway of gateways) {
      gateway.isCN = this.hosts[gateway.router] && this.hosts[gateway.router].location.startsWith('CN');
      for (const boolField of ['hidden', 'redirectAllPorts', 'noOut', 'badUdp']) {
        gateway[boolField] = !!gateway[boolField]
      }
      if (gateway.mark) {
        gateway.selectionMark = gateway.mark + 50;
        gateway.redirectServerPort = gateway.mark + 60000;
      } else {
        gateway.selectionMark = 0;
        gateway.redirectServerPort = 60100;
      }
      gateway.redirectTargetPorts = (gateway.redirectAllPorts || gateway.tunProxy) ? '1:65535' : this.vars.redirectTargetPorts.replace(/-/g, ':');
      gateway.pccRules = [];
    }
    return gateways;
  }

  async main() {
    this.hosts = _.keyBy(await this.load('nextgen2'), 'name');
    const subnets = await this.load('subnets');
    for (const [name, router] of Object.entries(this.hosts)) {
      router.lanInterfaces = subnets
        .filter((subnet) => subnet.router === name)
        .map((subnet) => subnet.interface)
        .filter((i) => i);
      router.subnets = subnets.filter((subnet) => subnet.router === name).map((subnet) => subnet.subnet);
    }
    this.vars = await this.loadUtilities();
    // @ts-ignore
    this.gateways = _.mapValues(_.groupBy(await this.loadGateways(), 'router'), g => _.keyBy(g, 'isp'));
    this.gatewayGroups = await this.load('gateway groups') as GatewayGroup[];
    //console.log(this.gateways);
    this.connections = _.intersection(Object.keys(this.hosts), Object.keys(_.find(this.hosts)));


    for (const host of Object.values(this.hosts)) {
      host.wgPublickey = await this.wgPublickey(host.wgPrivateKey);
    }
    if (process.env.ONLY_LINKS) {
      this.linksOnly = process.env.ONLY_LINKS.split(",");
    }
    if (process.env.LIMIT_LINKS) {
      this.linksLimit = process.env.LIMIT_LINKS.split(",");
    }
    if (process.env.ONLY_PROTOS) { 
      this.protosOnly = process.env.ONLY_PROTOS.split(",");
    }
    const inventoryValue = { wg: { hosts: Object.fromEntries(Object.values(this.hosts).map(host => [host.name, this.getHostConnectionInfo(host)])) } };
    await fs.promises.writeFile('result/inventory.yaml', YAML.stringify(inventoryValue));
    // console.log(Object.values(this.hosts));
    const hosts = await Promise.all(Object.values(this.hosts).map(async (h) => ({
      name: h.name,
      vars: await this.host_vars(h)
    })));
    //const hosts = Object.fromEntries(hostsArray);
    //await fs.promises.writeFile('result/inventory.yaml', YAML.stringify({
    //  wg: { hosts }
    //}));
    await fs.promises.writeFile('result/global-vars.yaml', YAML.stringify(this.vars));
    await Promise.all(hosts.map(host => fs.promises.writeFile(`result/vars-${host.name}.yaml`, YAML.stringify(host.vars))));
  }

  async loadUtilities() {
    const raw_utility = _.keyBy(await this.load('configurations'), 'key');
    this.routeLists = YAML.parse(fs.readFileSync(path.join('lists', 'result.yaml'), 'utf8'));
    // 所有内网网段
    this.routeLists.mycard = ['10.198.0.0/16', '10.200.0.0/15', '172.16.0.0/12'];
    this.routeLists.whitelisted = ['10.198.0.0/16']
    for (const h in this.hosts) {
      const host = this.hosts[h];
      for (const c of host.subnets) {
        if (!c.length) {
          continue;
        }
        this.routeLists.mycard.push(c);
        this.routeLists.whitelisted.push(c);
      }
    }
    const vars = {
      routeLists: this.routeLists,
      routeListNames: Object.keys(this.routeLists),
      noUpdateLinks: !!process.env.NO_LINK
    };
    for (let col in raw_utility) {
      vars[col] = raw_utility[col].value;
    }

    return vars;
  }
  getRoutePlanAddressesString(addresses: string[]) {
    if (!addresses.length) {
      return null;
    }
    return addresses.join(" ");
  }
  isGatewayGroupContains(gatewayGroup: GatewayGroup, host: CommonEntry) {
    const locationPrefixes = gatewayGroup.locationPrefix.split(",");
    const excludeRouters = gatewayGroup.excludeRouters.split(",");
    const includeRouters = gatewayGroup.includeRouters.split(",");
    const children = gatewayGroup.children.split(",");
    if (excludeRouters.includes(host.name)) {
      return false;
    }
    if (locationPrefixes.some(prefix => prefix !== "" && (host.location as string).startsWith(prefix)) || includeRouters.includes(host.name)) {
      return true;
    }
    for (let childName of children) {
      const targetGatewayGroup = this.gatewayGroups.find(g => g.name === childName);
      if (!targetGatewayGroup) {
        continue;
      }
      if (this.isGatewayGroupContains(targetGatewayGroup, host)) {
        return true;
      }
    }
    return false;
  }
  getAddressesFromGatewayGroup(gatewayGroup: GatewayGroup, hosts: CommonEntry[]) {
    const suitableHosts = hosts.filter(host => this.isGatewayGroupContains(gatewayGroup, host));
    return suitableHosts.map(host => host.address);
  }
  getRoutePlansFromGatewayGroups(host: CommonEntry) {
    const allOtherHosts = Object.values(this.hosts).filter(h => h !== host.name)
    const routePlans = this.gatewayGroups.filter(group => !this.isGatewayGroupContains(group, host)).map(group => {
      const addresses = this.getAddressesFromGatewayGroup(group, allOtherHosts);
      return {
        name: group.name.replace(/-/g, "_"),
        destMark: group.destMark,
        addresses,
        addressesString: this.getRoutePlanAddressesString(addresses)
      }
    }).filter(plan => plan.addresses.length > 0);
    return routePlans;
  }

  getHostConnectionInfo(host: CommonEntry) {
    return {
      ansible_ssh_host: host.host,
      ansible_ssh_user: host.user,
      ansible_ssh_port: host.sshPort || 22,
      ansible_python_interpreter: host.python || 'python3',
      noBird: !!host.noBird,
      systemBird: !!host.sysBird
    }
  }

  async host_vars(host: CommonEntry) {
    const connections = [];
    host.dockerServices = {
      version: '2.4',
      services: {
        'gateways-monitor': {
          restart: 'always',
          image: `git-registry.mycard.moe/railgun/gateways-monitor${this.getDockerImageTag(host)}`,
          network_mode: 'host',
          cap_add: ['NET_ADMIN'],
          volumes: ['./route-plans:/usr/src/app/route-plans:ro'],
          command: '/usr/src/app/gateway-monitor.sh'
        },
      }
    };
    if (!host.sysBird) {
      host.dockerServices.services.babeld = {
        restart: 'always',
        image: `git-registry.mycard.moe/railgun/babeld${this.getDockerImageTag(host)}`,
        network_mode: 'host',
        //cap_add: ['NET_ADMIN'],
        privileged: true,
        volumes: ['./babeld.conf:/etc/babeld.conf:ro']
      };
    }

    const availableGateways = Object.values(this.gateways[host.name]).filter(gateway => !gateway.hidden);

    const gostConfig: GostConfig = {
      Routes: availableGateways.map(gateway => {
        let tunProxy: string = gateway.tunProxy;
        if (gateway.tunProxy === 'warp') {
          tunProxy = `http://localhost:${gateway.redirectServerPort - 2000}`;
        }
        return {
          ServeNodes: [`red://${host.address}:${gateway.redirectServerPort}`],
          Mark: tunProxy?.includes('localhost') ? 0 : gateway.selectionMark as number,
          ChainNodes: tunProxy ? tunProxy.split('|') : undefined
        }
      })
    };

    const allRedirectServerPorts = availableGateways.map(gateway => gateway.redirectServerPort).join(',');

    if (gostConfig.Routes.length) {
      host.gostConfig = gostConfig;
      host.dockerServices.services.gost = {
        restart: 'always',
        image: `git-registry.mycard.moe/nanahira/gost${this.getDockerImageTag(host)}`,
        network_mode: 'host',
        privileged: true,
        volumes: ['./gost.json:/etc/gost/gost.json:ro'],
        command: '-C /etc/gost/gost.json'
      };
      if (host.arch !== 'arm') {
        for (const gateway of availableGateways.filter(gw => gw.tunProxy === 'warp')) {
          host.dockerServices.services[`warp-${gateway.isp}`] = {
            restart: 'always',
            image: 'git-registry.mycard.moe/nanahira/warp-proxy',
            command: 'sleep infinity',
            network_mode: 'host',
            environment: {
              LOCAL_PROXY_PORT: gateway.redirectServerPort - 2000
            }
          }
        }
      }
    }

    const pccGateways = availableGateways.filter(gateway => gateway.pccWeight);
    if (pccGateways.length) { 
      const gwPccRules = getRules(pccGateways.map(gw => gw.pccWeight));
      gwPccRules.forEach((rule, index) => pccGateways[index].pccRules = rule);
    }
    
    host.frpcRestarts = [];
    host.ocRestarts = [];
    host.frpsNeeded = false;
    const null_connection = '10000,null';
    const lanInterfaces = host.lanInterfaces;
    const localSubnets = host.subnets;
    //console.log(localSubnets);
    const masqInterfaces = host.masqInterfaces.length > 0 ? (host.masqInterfaces as string).split(',').map((interfaceName, interfaceIndex) => {
      return {
        name: interfaceName,
        mark: 1900 + interfaceIndex
      }
    }) : [];
    const routePlans = this.getRoutePlansFromGatewayGroups(host);

    for (const h of this.connections) {
      if (h != host.name) {
        const to = host[h]; // 当前主机的条目
        const from = this.hosts[h][host.name]; // 其他主机的这个主机的条目
        if (from && to) {
          // 非对称连接
          connections.push(await this.parse_connection(host, this.hosts[h], to, false, true, false));
          connections.push(await this.parse_connection(host, this.hosts[h], from, true, false, true));
        } else if (from || to) {
          // 对称连接
          const connectionString = from || to;
          connections.push(await this.parse_connection(host, this.hosts[h], connectionString, true, true, connectionString === from));
          connections.push(await this.parse_connection(host, this.hosts[h], null_connection, false, false, false));
        } else {
          // 不连接
          connections.push(await this.parse_connection(host, this.hosts[h], null_connection, true, false, false));
          connections.push(await this.parse_connection(host, this.hosts[h], null_connection, false, true, false));
        }
        const targetHost = this.hosts[h];
        routePlans.push({
          name: h.replace(/-/g, "_"),
          destMark: targetHost.destMark,
          addresses: [targetHost.address],
          addressesString: this.getRoutePlanAddressesString([targetHost.address])
        });
      }
    }

    return {
      //ansible_ssh_host: host.host,
      //ansible_ssh_user: host.user,
      //ansible_ssh_port: host.sshPort || 22,
      //ansible_python_interpreter: host.python || 'python3',
      id: host.id,
      address: host.address,
      isCN: host.location.startsWith('CN'),
      key: host.wgPrivateKey,
      frpsNeeded: host.frpsNeeded,
      frpsPort: host.frpsPort,
      ocRestarts: host.ocRestarts.length ? host.ocRestarts : false,
      frpcRestarts: host.frpcRestarts.length ? host.frpcRestarts : false,
      ocservNeeded: host.ocservNeeded || false,
      ocservPort: host.ocservPort,
      ocservCert: host.ocservCert || null,
      ocMetric: host.ocMetric || null,
      ocpasswdLines: host.ocpasswdLines || [],
      gateways: _.values(this.gateways[host.name]),
      connections,
      lanInterfaces,
      localSubnets,
      masqInterfaces,
      dockerServices: host.dockerServices,
      routePlans,
      iptables_type: host.iptables || 'auto',
      gostConfig,
      installGost: !!gostConfig,
      allRedirectServerPorts,
    };
  }

  makeV6Domain(domain: string) {
    const domainParts = domain.split('.');
    domainParts[domainParts.length - 3] = domainParts[domainParts.length - 3].replace('-v4', '-v6');
    return domainParts.join('.');
  }

  async parse_connection(local: any, remote: any, connstr: string, inbound: boolean, outbound: boolean, reverse: boolean) {
    const leftbottom = local.id > remote.id; // true 条目位于左下，false 条目位于右上
    const cis = !reverse; // true 无需翻转，false 需要翻转。
    const primary = leftbottom ? outbound : inbound; // true 使用 peerAddress、port, false 使用peerAddress2、port2
    const connStrSplited = connstr.split(',');
    const [_metric, protocol] = connStrSplited;
    const paramsString = connStrSplited.slice(2).join('&');
    const metric = parseInt(_metric);
    const params = Object.fromEntries(new URLSearchParams(paramsString).entries());
    const name = `mc${!outbound ? 'i' : '-'}${remote.name}`;
    const remoteName = `mc${!inbound ? 'i' : '-'}${local.name}`;
    const localGatewayName = (cis ? params.lif : params.rif) || params.if;
    const localGateway = localGatewayName ? this.gateways[local.name][localGatewayName] : _.find(this.gateways[local.name]);
    assert(localGateway, `Gateway ${localGatewayName} for ${local.name} not found.`);
    //console.log(local.name, paramsString, params, localGatewayName, localGateway.name)
    const localGatewayMark = localGateway.mark || 0;
    const remoteGatewayName = (cis ? params.rif : params.lif) || params.if;
    const remoteGateway = remoteGatewayName ? this.gateways[remote.name][remoteGatewayName] : _.find(this.gateways[remote.name]);
    assert(remoteGateway, `Gateway ${remoteGatewayName} for ${remote.name} not found.`);
    //const remoteGatewayMark = remoteGatewayName ? remoteGateway.mark : undefined;
    //console.log(remoteGateway.name);
    let remoteAddress = remoteGateway.address || null;
    const preferV6 = (!remoteGateway.ipv4 || params.ipv6 === 'on' || params.ipv6 === 'force') && remoteGateway.ipv6 && params.ipv6 !== 'off';
    if (!localGateway.ipv6 && preferV6 || localGateway.noOut) {
      remoteAddress = null;
    }
    if (remoteAddress && params.ipv6 === 'force') {
      remoteAddress = this.makeV6Domain(remoteAddress);
    }
    const resolvedRemoteAddress = remoteAddress ? await this.resolveDomain(remoteAddress, preferV6) : null;
    const remoteLocalAddress = remote.address;
    const remoteNextMark = remote.nextMark;
    const remoteDestMark = remote.destMark;
    const localPort = (primary ? remote.port : remote.port2) + local.offset;
    const remotePort = (primary ? local.port : local.port2) + remote.offset;
    const remoteFrpsPort = remote.frpsPort;
    const remoteOcservPort = remote.ocservPort;
    const wgPublicKey = remote.wgPublickey;
    const localPeerAddress = primary ? `10.200.${local.id}.${remote.id}` : `10.201.${local.id}.${remote.id}`;
    const remotePeerAddress = primary ? `10.200.${remote.id}.${local.id}` : `10.201.${remote.id}.${local.id}`;
    const localPeerAddress6Block = ((local.id << 8) | remote.id).toString(16);
    const remotePeerAddress6Block = ((remote.id << 8) | local.id).toString(16);

    const localPeerAddress6 = `fe80::${primary ? 1 : 2}:${localPeerAddress6Block}`;
    const remotePeerAddress6 = `fe80::${primary ? 1 : 2}:${remotePeerAddress6Block}`;

    const frpType = protocol === 'wgfrp' ? (this.gatewayCompare(localGateway, remoteGateway) ? 'frps' : 'frpc') : undefined;
    const ocType = protocol === 'oc' ? (this.gatewayCompareOcserv(local, remote, localGateway, remoteGateway) ? 'server' : 'client') : undefined;

    const noUpdate = this.linksOnly && !(this.linksOnly.includes(remote.name) || this.linksOnly.includes(local.name)) || this.linksLimit && !(this.linksLimit.includes(remote.name) && this.linksLimit.includes(local.name)) || this.protosOnly && !this.protosOnly.includes(protocol);

    if (frpType === 'frps' && !local.dockerServices.services.frps) {
      local.frpsNeeded = true;
      local.dockerServices.services.frps = {
        restart: 'always',
        image: 'fatedier/frps:v0.34.2',
        network_mode: 'host',
        command: '-c /frps.ini',
        volumes: ['./frps.ini:/frps.ini:ro']
      };
    }

    if (frpType === 'frpc') {
      const containerName = `frpc-${name}`;
      local.dockerServices.services[containerName] = {
        restart: 'always',
        image: 'fatedier/frpc:v0.34.2',
        network_mode: 'host',
        command: '-c /frpc.ini',
        volumes: [`./frpc-${name}.ini:/frpc.ini:ro`]
      };
      if (!noUpdate) {
        local.frpcRestarts.push(containerName);
      }
    }

    if (ocType === 'server') {
      if (!local.dockerServices.services.ocserv) {
        local.ocservNeeded = true;
        local.ocpasswdLines = [];
        local.ocMetric = metric;
        local.dockerServices.services.ocserv = {
          restart: 'always',
          image: `git-registry.mycard.moe/nanahira/docker-ocserv${this.getDockerImageTag(local)}`,
          network_mode: 'host',
          command: 'ocserv -f -d 1',
          cap_add: ['NET_ADMIN'],
          devices: ['/dev/net/tun:/dev/net/tun'],
          volumes: [
            './ocserv/ocserv.conf:/etc/ocserv/ocserv.conf:ro',
            './ocserv/config-per-user:/etc/ocserv/config-per-user:ro',
            './ocserv/env-per-user:/etc/ocserv/env-per-user:ro',
            './ocserv/ocpasswd:/etc/ocserv/ocpasswd:ro',
            `./ocserv/certs/${local.ocservCert}:/etc/ssl/certs/${local.ocservCert}:ro`,
            '$HOME/nextgen-network/scripts:$HOME/nextgen-network/scripts:ro',
          ]
        };
        if (!local.sysBird) {
          local.dockerServices.services.ocserv.depends_on = ['babeld'];
        }
      }
      local.ocpasswdLines.push(await generateOcpasswdLine(name, this.vars.ocservPassword));
    }

    const mtu = Math.min(localGateway ? localGateway.mtu : 1500, remoteGateway ? remoteGateway.mtu : 1500);

    if (ocType === 'client') {
      let startupCommand = `echo "${this.vars.ocservPassword}" | openconnect --user=${remoteName} --passwd-on-stdin --passtos --interface=${name} --mtu=${mtu} ${remoteAddress}:${remoteOcservPort}`;
      if (params.p === "tcp" || remoteGateway.badUdp || localGateway.badUdp) {
        startupCommand += ' --no-dtls';
      }
      const containerName = `openconnect-${name}`;
      local.dockerServices.services[containerName] = {
        restart: 'always',
        image: `git-registry.mycard.moe/railgun/openconnect${this.getDockerImageTag(local)}`,
        network_mode: 'host',
        command: ['bash', '-c', startupCommand],
        cap_add: ['NET_ADMIN'],
        devices: ['/dev/net/tun:/dev/net/tun'],
        environment: {
          'FORCE_RELOAD': '2020.1.11'
        },
        volumes: [
          `./client-scripts/${name}:/etc/vpnc:ro`,
          '$HOME/nextgen-network/scripts:$HOME/nextgen-network/scripts:ro'
        ]
      };
      if (!noUpdate) {
        local.ocRestarts.push(containerName);
      }
    }

    //console.log(local.name, name, mtu);

    if (outbound) {
      console.log(`${local.name} GW ${localGateway.isp} ${inbound ? '<' : '='}=${(frpType === 'frps' || ocType === 'server') ? 's' : '='}=[${protocol}]=${(frpType === 'frpc' || ocType === 'client') ? 's' : '='}=> ${remote.name} GW ${remoteGateway.isp}`);
    }

    return {
      name,
      metric,
      protocol,
      params,
      localGatewayMark,
      remoteNextMark,
      remoteDestMark,
      remoteAddress,
      resolvedRemoteAddress,
      remoteLocalAddress,
      localPort,
      remotePort,
      wgPublicKey,
      localPeerAddress,
      remotePeerAddress,
      localPeerAddress6,
      remotePeerAddress6,
      remoteFrpsPort,
      //remoteOcservPort,
      frpType,
      ocType,
      inbound,
      outbound,
      mtu,
      noUpdate,
    };
  }

  // frps还是frpc的积分，NAT越有利分越高
  gatewayCompareScore(gateway: any): number {
    let offset = 0;
    let score = 0;
    score |= (0xff - gateway.id) << offset; // 8 bits
    offset += 8;
    const isCNScore = gateway.isCN ? 0 : 1; // 1 bit
    score |= isCNScore << offset;
    offset += 1;
    const ipv4NatScore = ({ // 2 bits
      'ports': 0,
      'dmz': 1
    })[gateway.ipv4Nat] || 2;
    score |= ipv4NatScore << offset;
    offset += 2;
    const ipv4Score = ({ // 2 bits
      'static': 2,
      'dynamic': 1
    })[gateway.ipv4] || 0;
    score |= ipv4Score << offset;
    offset += 2;
    const noInScore = gateway.noOut ? 1 : 0; // 1 bit
    score |= noInScore << offset;
    offset += 1;
    return score;
  }

  // true: 本地做 server，false: 远端做server
  // 如果都不能做，抛异常
  // 两个参数对调返回的结果必须相反
  gatewayCompare(localGateway: any, remoteGateway: any): boolean {
    // 两边至少一个有 IPv4 地址，或两边都有 IPv6 地址才能连
    assert(localGateway.ipv4 !== '' || remoteGateway.ipv4 !== '' || (localGateway.ipv6 !== '' && remoteGateway.ipv6 !== ''));
    // 两边不能都是只能入站的
    assert(!localGateway.noOut || !remoteGateway.noOut);
    
    const localScore = this.gatewayCompareScore(localGateway);
    const remoteScore = this.gatewayCompareScore(remoteGateway);
    // 两边必须相反
    assert(localScore !== remoteScore);
    // 两边只有一边是GlobalSSH的，GlobalSSH做s
    // 只有一边有static的，就static做s
    // 如果都static，那么没有NAT的或者dmz的做s
    // 如果都还相同，就让列表中更靠前的做s
    return localScore > remoteScore;
  }

  gatewayCompareOcserv(local: any, remote: any, localGateway: any, remoteGateway: any): boolean {
    // 两边至少一个有证书才能连
    assert(local.ocservCert || remote.ocservCert);
    // 只有一边有证书用有证书的那边
    if (!!local.ocservCert !== !!remote.ocservCert) {
      return !!local.ocservCert;
    }

    return this.gatewayCompare(localGateway, remoteGateway);
  }

  async wgPublickey(privateKey) {
    return new Promise((resolve, reject) => {
      const child = child_process.execFile('wg', ['pubkey'], { encoding: 'utf8' }, (error, stdout, stderr) => {
        if (stderr) {
          console.warn(stderr);
        }
        if (error) {
          reject(error);
        } else {
          resolve(stdout.trimEnd());
        }
      });
      child.stdin.end(privateKey);
    });
  }
}

new InventoryBuilder().main();

// export interface Host {
//
// }

// export interface Interface {
//
// }
