mod router;

use crate::router::{Router, RouterReader, RouterWriter, SECRET_LENGTH};
use std::collections::HashMap;
use std::env;
use std::error::Error;
use std::intrinsics::transmute;
use std::io::{Read, Write};
use std::mem::MaybeUninit;
use std::sync::Arc;

#[repr(C)]
pub struct Meta {
    pub src_id: u8,
    pub dst_id: u8,
    pub reversed: u16,
}

use serde::{Deserialize, Serialize};

#[derive(Debug, Serialize, Deserialize)]
pub struct ConfigRouter {
    pub remote_id: u8,
    pub proto: i32,
    pub family: u8,
    pub mark: u32,
    pub endpoint: String,
    pub remote_secret: String,
    pub dev: String,
    pub up: String,
}

#[derive(Debug, Serialize, Deserialize)]
pub struct Config {
    pub local_id: u8,
    pub local_secret: String,
    pub routers: Vec<ConfigRouter>,
}
use crossbeam_utils::thread;
use grouping_by::GroupingBy;
use lazy_static::lazy_static;
use pnet::packet::ipv4::Ipv4Packet;
use socket2::Socket;

lazy_static! {
    static ref config: Config = serde_json::from_str(env::args().nth(0).unwrap().as_str()).unwrap();
    static ref local_secret: [u8; SECRET_LENGTH] =
        Router::create_secret(config.local_secret.as_str()).unwrap();
}

fn main() -> Result<(), Box<dyn Error>> {
    let mut sockets: HashMap<u16, Arc<Socket>> = HashMap::new();
    let routers: HashMap<u8, Router> = config
        .routers
        .iter()
        .map(|c| Router::new(c, &mut sockets).map(|router| (c.remote_id, router)))
        .collect::<Result<_, _>>()?;
    let (mut router_readers, router_writers): (
        HashMap<u8, RouterReader>,
        HashMap<u8, RouterWriter>,
    ) = routers
        .into_iter()
        .map(|(id, router)| {
            let (reader, writer) = router.split();
            ((id, reader), (id, writer))
        })
        .unzip();

    let router_writers3: Vec<(Arc<Socket>, HashMap<u8, RouterWriter>)> = router_writers
        .into_iter()
        .grouping_by(|(_, v)| v.key())
        .into_iter()
        .map(|(k, v)| {
            (
                Arc::clone(sockets.get_mut(&k).unwrap()),
                v.into_iter().collect(),
            )
        })
        .collect();

    thread::scope(|s| {
        for router in router_readers.values_mut() {
            s.spawn(|_| {
                let mut buffer = [0u8; 1500 - 20]; // minus typical IP header space
                let meta_size = size_of::<Meta>();

                // Pre-initialize with our Meta header (local -> remote)
                let meta = Meta {
                    src_id: config.local_id,
                    dst_id: router.config.remote_id,
                    reversed: 0,
                };
                // Turn the Meta struct into bytes
                let meta_bytes = unsafe {
                    std::slice::from_raw_parts(&meta as *const Meta as *const u8, meta_size)
                };
                buffer[..meta_size].copy_from_slice(meta_bytes);

                loop {
                    let n = router.tun_reader.read(&mut buffer[meta_size..]).unwrap();
                    if let Some(ref addr) = *router.endpoint.read().unwrap() {
                        router.encrypt(&mut buffer[meta_size..meta_size + n]);
                        #[cfg(target_os = "linux")]
                        router.socket.set_mark(router.config.mark);
                        let _ = router.socket.send_to(&buffer[..meta_size + n], addr);
                    }
                }
            });
        }

        for (socket, mut router_writers) in router_writers3 {
            s.spawn(move |_| {
                let mut recv_buf = [MaybeUninit::uninit(); 1500];
                loop {
                    let _ = (|| {
                        let (len, addr) = socket.recv_from(&mut recv_buf).unwrap();
                        let data: &mut [u8] = unsafe { transmute(&mut recv_buf[..len]) };

                        let packet = Ipv4Packet::new(data).ok_or("malformed packet")?;
                        let header_len = packet.get_header_length() as usize * 4;
                        let (_ip_header, rest) = data.split_at_mut_checked(header_len).ok_or("malformed packet")?;
                        let (meta_bytes, payload) = rest.split_at_mut_checked(size_of::<Meta>()).ok_or("malformed packet")?;
                        let meta: &Meta = unsafe { transmute(meta_bytes.as_ptr()) };
                        if meta.dst_id == config.local_id && meta.reversed == 0 {
                            let router = router_writers.get_mut(&meta.src_id).ok_or("missing router")?;
                            *router.endpoint.write().unwrap() = Some(addr);
                            router.decrypt(payload);
                            router.tun_writer.write_all(payload)?;
                        }

                        Ok::<(), Box<dyn Error>>(())
                    })();
                }
            });
        }
    })
    .unwrap();
    Ok(())
}
