use socket2::{Domain, Protocol, SockAddr, Socket, Type};
use std::collections::hash_map::Entry;
use std::collections::HashMap;
use std::net::ToSocketAddrs;
use std::process::Command;
use std::sync::Arc;
use tun::{Reader, Writer};
pub const SECRET_LENGTH: usize = 32;
use crate::ConfigRouter;
use base64::prelude::*;

pub struct Router {
    pub config: &'static ConfigRouter,
    pub secret: [u8; SECRET_LENGTH],
    pub endpoint: Option<SockAddr>,
    pub tun_reader: Reader,
    pub tun_writer: Writer,
    pub socket: Arc<Socket>,
}

impl Router {
    pub(crate) fn create_secret(
        config: &str,
    ) -> Result<[u8; SECRET_LENGTH], Box<dyn std::error::Error>> {
        let mut secret = [0u8; SECRET_LENGTH];
        let decoded = BASE64_STANDARD.decode(config)?;
        let len = decoded.len().min(SECRET_LENGTH);
        secret[..len].copy_from_slice(&decoded[..len]);
        Ok(secret)
    }
    fn create_raw_socket(
        config: &ConfigRouter,
        sockets: &mut HashMap<u16, Arc<Socket>>,
    ) -> Result<Arc<Socket>, Box<dyn std::error::Error>> {
        let key = (config.family as u16) << 8 | config.proto as u16;
        let result = match sockets.entry(key) {
            Entry::Occupied(entry) => entry.get().clone(),
            Entry::Vacant(entry) => entry
                .insert(Arc::new(Socket::new(
                    if config.family == 6 {
                        Domain::IPV6
                    } else {
                        Domain::IPV4
                    },
                    Type::RAW,
                    Some(Protocol::from(config.proto)),
                )?))
                .clone(),
        };
        Ok(result)
    }
    fn create_tun_device(
        config: &ConfigRouter,
    ) -> Result<(Reader, Writer), Box<dyn std::error::Error>> {
        let mut tun_config = tun::Configuration::default();
        tun_config.tun_name(config.dev.as_str()).up();

        let dev = tun::create(&tun_config)?;
        Ok(dev.split())
    }
    fn run_up_script(config: &ConfigRouter) -> Result<(), Box<dyn std::error::Error>> {
        Command::new(config.up.as_str()).status()?;
        Ok(())
    }

    fn create_endpoint(
        config: &ConfigRouter,
    ) -> Result<Option<SockAddr>, Box<dyn std::error::Error>> {
        let parsed = config.endpoint.to_socket_addrs()?.next().unwrap();
        Ok(Some(parsed.into()))
    }

    pub fn new(
        config: &'static ConfigRouter,
        sockets: &mut HashMap<u16, Arc<Socket>>,
    ) -> Result<Router, Box<dyn std::error::Error>> {
        let secret = Self::create_secret(config.remote_secret.as_str())?;
        let endpoint = Self::create_endpoint(&config)?;
        let socket = Self::create_raw_socket(&config, sockets)?;
        let (tun_reader, tun_writer) = Self::create_tun_device(&config)?;
        Self::run_up_script(&config)?;

        let router = Router {
            config,
            secret,
            endpoint,
            tun_reader,
            tun_writer,
            socket,
        };

        Ok(router)
    }

    pub(crate) fn encrypt(&self, data: &mut [u8]) {
        for (i, b) in data.iter_mut().enumerate() {
            *b ^= self.secret[i % SECRET_LENGTH];
        }
    }
    pub(crate) fn decrypt(&self, data: &mut [u8]) {
        self.encrypt(data);
    }
}
