import parse from 'csv-parse';
import util from 'util';
import fs from 'fs';
import path from 'path';
import YAML from 'yaml';
import _ from 'lodash';

class InventoryBuilder {
  hosts: { [key: string]: any };
  gateways: any;
  connections: string[];

  async load(sheetName) {
    const data = await fs.promises.readFile(path.join('data', `内网互联计划 - ${sheetName}.csv`));
    // @ts-ignore
    return (await util.promisify(parse)(data, { columns: true, cast: true })).filter(h => h.id);
  }

  async main() {
    this.hosts = _.keyBy(await this.load('nextgen links'), 'name');
    this.gateways = _.mapValues(_.groupBy(await this.load('gateways'), 'name'), g => _.keyBy(g, 'isp'));
    this.connections = _.intersection(Object.keys(this.hosts), Object.keys(_.find(this.hosts)));

    console.log(Object.values(this.hosts));
    const hosts = Object.fromEntries(Object.values(this.hosts).map(h => [h.host, this.host_vars(h)]));
    console.log(hosts);
    const result = YAML.stringify({ wg: { hosts } });
    return fs.promises.writeFile('result/inventory.yaml', result);
  }

  host_vars(host) {
    const connections = [];
    for (const h of this.connections) {
      if (h != host.name) {
        const to = host[h];
        const from = this.hosts[h][host.name];
        if (from && to) {
          // 非对称连接
          connections.push(this.parse_connection(h, from, true));
          connections.push(this.parse_connection(h, to));
        } else if (from || to) {
          // 对称连接
          connections.push(this.parse_connection(h, from || to));
        }
        // 不连接
      }
    }

    return {
      ansible_ssh_user: host.user,
      address: host.address,
      key: host.wgPrivateKey,
      gateways: _.mapValues(this.gateways[host.name], gw => _.pick(gw, ['mark_gateway'])),
      connections
    };
  }

  parse_connection(_name: string, str: string, inbound = false) {
    const [_metric, protocol, _params] = str.split(',');
    const metric = parseInt(_metric);
    const params = Object.fromEntries(new URLSearchParams(_params).entries());
    const mark = this.hosts[_name].mark;
    const name = inbound ? `${_name}-in` : _name;
    return { name, metric, protocol, params, mark, inbound };
  }
}

new InventoryBuilder().main();
