mod router;

use crate::router::{Router, RouterReader, RouterWriter};
use std::collections::HashMap;
use std::env;
use std::error::Error;
use std::intrinsics::transmute;
use std::io::{Read, Write};
use std::mem::MaybeUninit;
use std::sync::Arc;

#[repr(C)]
pub struct Meta {
    pub src_id: u32,
    pub dst_id: u32,
    pub reversed: u16,
}

use serde::Deserialize;

#[derive(Deserialize)]
pub struct ConfigRouter {
    pub local_id: u32,
    pub remote_id: u32,
    pub proto: i32,
    pub family: u8,
    pub mark: u32,
    pub endpoint: String,
    pub local_secret: String,
    pub remote_secret: String,
    pub dev: String,
    pub up: String,
}

#[derive(Deserialize)]
pub struct Config {
    pub routers: Vec<ConfigRouter>,
}
use crossbeam_utils::thread;
use grouping_by::GroupingBy;
use pnet::packet::ipv4::Ipv4Packet;
use socket2::Socket;
use std::fs;

fn main() -> Result<(), Box<dyn Error>> {
    println!("Init");
    let args: Vec<String> = env::args().collect();
    if args.len() < 2 {
        return Err("need JSON string or -c <config.json>".into());
    }

    let config: Config;

    if args[1] == "-c" || args[1] == "--config" {
        // 从文件读
        if args.len() < 3 {
            return Err("missing value for -c/--config".into());
        }
        let data = fs::read_to_string(&args[2])?;
        config = serde_json::from_str(&data)?;
    } else {
        // 当作 JSON 字符串解析
        config = serde_json::from_str(&args[1])?;
    }
    println!("Read config");
    let mut sockets: HashMap<u16, Arc<Socket>> = HashMap::new();
    println!("Ready");
    let routers: HashMap<u32, Router> = config
        .routers
        .iter()
        .map(|c| Router::new(c, &mut sockets).map(|router| (c.remote_id, router)))
        .collect::<Result<_, _>>()?;
    println!("Created routers");
    let (mut router_readers, router_writers): (
        HashMap<u32, RouterReader>,
        HashMap<u32, RouterWriter>,
    ) = routers
        .into_iter()
        .map(|(id, router)| {
            let (reader, writer) = router.split();
            ((id, reader), (id, writer))
        })
        .unzip();
    println!("Created sockets");
    let router_writers3: Vec<(Arc<Socket>, HashMap<u32, RouterWriter>)> = router_writers
        .into_iter()
        .grouping_by(|(_, v)| v.key())
        .into_iter()
        .map(|(k, v)| {
            (
                Arc::clone(sockets.get_mut(&k).unwrap()),
                v.into_iter().collect(),
            )
        })
        .collect();
    println!("created tuns");

    thread::scope(|s| {
        for router in router_readers.values_mut() {
            s.spawn(|_| {
                let mut buffer = [0u8; 1500 - 20]; // minus typical IP header space
                let meta_size = size_of::<Meta>();

                // Pre-initialize with our Meta header (local -> remote)
                let meta = Meta {
                    src_id: router.config.local_id,
                    dst_id: router.config.remote_id,
                    reversed: 0,
                };
                // Turn the Meta struct into bytes
                let meta_bytes = unsafe {
                    std::slice::from_raw_parts(&meta as *const Meta as *const u8, meta_size)
                };
                buffer[..meta_size].copy_from_slice(meta_bytes);

                loop {
                    let n = router.tun_reader.read(&mut buffer[meta_size..]).unwrap();
                    if let Some(ref addr) = *router.endpoint.read().unwrap() {
                        router.encrypt(&mut buffer[meta_size..meta_size + n]);
                        let _ = router.socket.send_to(&buffer[..meta_size + n], addr);
                    }
                }
            });
        }

        for (socket, mut router_writers) in router_writers3 {
            s.spawn(move |_| {
                let mut recv_buf = [MaybeUninit::uninit(); 1500];
                loop {
                    let _ = (|| {
                        let (len, addr) = socket.recv_from(&mut recv_buf).unwrap();
                        let data: &mut [u8] = unsafe { transmute(&mut recv_buf[..len]) };

                        let packet = Ipv4Packet::new(data).ok_or("malformed packet")?;
                        let header_len = packet.get_header_length() as usize * 4;
                        let (_ip_header, rest) = data
                            .split_at_mut_checked(header_len)
                            .ok_or("malformed packet")?;
                        let (meta_bytes, payload) = rest
                            .split_at_mut_checked(size_of::<Meta>())
                            .ok_or("malformed packet")?;
                        let meta: &Meta = unsafe { transmute(meta_bytes.as_ptr()) };
                        if meta.reversed == 0 {
                            let router = router_writers
                                .get_mut(&meta.src_id)
                                .ok_or("missing router")?;
                            if meta.dst_id == router.config.local_id {
                                *router.endpoint.write().unwrap() = Some(addr);
                                router.decrypt(payload);
                                router.tun_writer.write_all(payload)?;
                            }
                        }

                        Ok::<(), Box<dyn Error>>(())
                    })();
                }
            });
        }
    })
    .unwrap();
    Ok(())
}
