use anyhow::{Result, bail, ensure};
use socket2::{Domain, Protocol, SockAddr, SockFilter, Socket, Type};
use std::net::Shutdown;
use std::sync::Arc;
use std::{
    ffi::c_void,
    mem::MaybeUninit,
    net::{Ipv4Addr, Ipv6Addr, SocketAddrV4, SocketAddrV6, ToSocketAddrs},
    ops::Range,
    os::fd::{AsRawFd, FromRawFd},
    process::{Command, ExitStatus},
    sync::atomic::Ordering,
};
use tun::Device;

use crate::config::{ConfigRouter, Schema};
use crossbeam::epoch::{Atomic, pin};
use libc::{
    BPF_ABS, BPF_B, BPF_IND, BPF_JEQ, BPF_JMP, BPF_K, BPF_LD, BPF_LDX, BPF_MSH, BPF_RET, BPF_W, MSG_FASTOPEN, SO_ATTACH_REUSEPORT_CBPF, SOL_SOCKET, setsockopt,
    sock_filter, sock_fprog, socklen_t,
};

pub const SECRET_LENGTH: usize = 32;
pub const META_SIZE: usize = size_of::<Meta>();

#[repr(C)]
#[derive(Debug, Clone, Copy, Default)]
pub struct Meta {
    pub src_id: u8,
    pub dst_id: u8,
    pub reversed: u16,
}
impl Meta {
    pub fn as_bytes(&self) -> &[u8; META_SIZE] {
        unsafe { &*(self as *const Meta as *const [u8; META_SIZE]) }
    }
    pub fn from_bytes(bytes: &[MaybeUninit<u8>; META_SIZE]) -> &Meta {
        unsafe { &*(bytes.as_ptr() as *const Meta) }
    }
}

pub struct Router {
    pub config: ConfigRouter,
    pub tun: Device,
    pub socket: Socket,
    pub endpoint: Atomic<SockAddr>,

    pub tcp_listener_connection: Atomic<Arc<Socket>>,
}

impl Router {
    pub(crate) fn decrypt(&self, data: &mut [u8], secret: &[u8; SECRET_LENGTH]) {
        for (i, b) in data.iter_mut().enumerate() {
            *b ^= secret[i % SECRET_LENGTH];
        }
    }
    pub(crate) fn decrypt2(&self, data: &mut [u8], secret: &[u8; SECRET_LENGTH], range: Range<usize>) {
        for (i, b) in data[range.clone()].iter_mut().enumerate() {
            *b ^= secret[(range.start + i) % SECRET_LENGTH];
        }
    }

    pub(crate) fn encrypt(&self, data: &mut [u8]) {
        for (i, b) in data.iter_mut().enumerate() {
            *b ^= self.config.remote_secret[i % SECRET_LENGTH];
        }
    }

    pub fn create_socket(config: &ConfigRouter, local_id: u8) -> Result<Socket> {
        match config.schema {
            Schema::IP => {
                let result = Socket::new(config.family, Type::RAW, Some(Protocol::from(config.proto as i32)))?;
                result.set_mark(config.mark)?;
                Self::attach_filter_ip(config, local_id, &result)?;
                Ok(result)
            }
            Schema::UDP => {
                let result = Socket::new(config.family, Type::DGRAM, Some(Protocol::UDP))?;
                result.set_mark(config.mark)?;
                if config.src_port != 0 {
                    result.set_reuse_port(true)?;
                    let addr = Self::bind_addr(config);
                    result.bind(&addr)?;
                }
                Ok(result)
            }
            Schema::TCP => {
                if config.dst_port == 0 {
                    // listener
                    let result = Socket::new(config.family, Type::STREAM, Some(Protocol::TCP))?;
                    let addr = Router::bind_addr(config);
                    result.bind(&addr)?;
                    result.listen(100)?;
                    Ok(result)
                } else {
                    // tcp client 初始化时不创建 socket，在循环里使用 connect_tcp 来创建
                    Ok(unsafe { Socket::from_raw_fd(0) })
                }
            }
        }
    }

    pub fn connect_tcp(&self, local_id: u8) -> Result<Socket> {
        // tcp client 的 socket 不要在初始化时创建，在循环里创建
        // 创建 socket 和 获取 endpoint 失败会 panic，连接失败会 error
        let result = Socket::new(self.config.family, Type::STREAM, Some(Protocol::TCP)).unwrap();
        result.set_tcp_nodelay(true).unwrap();
        result.set_mark(self.config.mark).unwrap();
        if self.config.src_port != 0 {
            result.set_reuse_address(true).unwrap();
            let addr = Self::bind_addr(&self.config);
            result.bind(&addr)?;
        }

        let meta = Meta {
            src_id: local_id,
            dst_id: self.config.remote_id,
            reversed: 0,
        };
        let guard = pin();
        let endpoint_ref = self.endpoint.load(Ordering::Relaxed, &guard);
        let endpoint = unsafe { endpoint_ref.as_ref() }.unwrap();

        result.send_to_with_flags(meta.as_bytes(), endpoint, MSG_FASTOPEN)?;
        Ok(result)
    }

    fn attach_filter_ip(config: &ConfigRouter, local_id: u8, socket: &Socket) -> Result<()> {
        // 由于多个对端可能会使用相同的 ipprpto 号，这里确保每个 socket 上只会收到自己对应的对端发来的消息
        let meta = Meta {
            src_id: config.remote_id,
            dst_id: local_id,
            reversed: 0,
        };
        let value = u32::from_be_bytes(*meta.as_bytes());

        // 如果你的协议是 UDP，这里必须是 8 (跳过 UDP 头: SrcPort(2)+DstPort(2)+Len(2)+Sum(2))
        // 如果是纯自定义 IP 协议，这里是 0
        let payload_offset = 0;

        // IP filter 工作原理：
        // 每个对端起一个 raw socket
        // 根据报文内容判断是给谁的。拒绝掉不是给自己的报文
        // IPv4 raw socket 带 IP 头，IPv6 不带

        let filters: &[SockFilter] = match socket.domain()? {
            Domain::IPV4 => &[
                // [IPv4] 计算 IPv4 头长度: X = 4 * (IP[0] & 0xf)
                bpf_stmt(BPF_LDX | BPF_B | BPF_MSH, 0),
                // A = Packet[X + payload_offset]
                bpf_stmt(BPF_LD | BPF_W | BPF_IND, payload_offset),
                // if (A == target_val) goto Accept; else goto Reject;
                bpf_jump(BPF_JMP | BPF_JEQ | BPF_K, value, 0, 1),
                // 【接受 (True 路径)】
                bpf_stmt(BPF_RET | BPF_K, u32::MAX),
                // 【拒绝 (False 路径)】
                bpf_stmt(BPF_RET | BPF_K, 0),
            ],
            Domain::IPV6 => &[
                // raw socket IPv6 没有 header，加载第 0 字节到累加器 A
                bpf_stmt(BPF_LD | BPF_W | BPF_ABS, 0),
                // if (A == target_val) goto Accept; else goto Reject;
                bpf_jump(BPF_JMP | BPF_JEQ | BPF_K, value, 0, 1),
                // 【接受 (True 路径)】
                bpf_stmt(BPF_RET | BPF_K, u32::MAX),
                // 【拒绝 (False 路径)】
                bpf_stmt(BPF_RET | BPF_K, 0),
            ],
            _ => bail!("unsupported family"),
        };
        socket.attach_filter(filters)?;

        Ok(())
    }

    pub fn attach_filter_udp(group: Vec<&Router>, local_id: u8) -> Result<()> {
        let values: Vec<u32> = group
            .iter()
            .map(|&f| {
                let meta = Meta {
                    src_id: f.config.remote_id,
                    dst_id: local_id,
                    reversed: 0,
                };
                u32::from_be_bytes(*meta.as_bytes())
            })
            .collect();

        let mut filters: Vec<SockFilter> = Vec::with_capacity(1 + values.len() * 2 + 1);

        // udp filter 工作原理：
        // 每个对端起一个 udp socket
        // 根据报文内容判断是给谁的，调度给对应的端口复用组序号

        // Load the first 4 bytes of the packet into the accumulator (A)
        filters.push(bpf_stmt(BPF_LD | BPF_W | BPF_ABS, 0));
        for (i, &val) in values.iter().enumerate() {
            // 如果匹配，继续下一句(返回)，如果不匹配，跳过下一句。
            filters.push(bpf_jump(BPF_JMP | BPF_JEQ | BPF_K, val, 0, 1));
            // If match, return the index (i + 1, since 0 means drop)
            filters.push(bpf_stmt(BPF_RET | BPF_K, i as u32));
        }
        // If no match found after all comparisons, drop the packet
        filters.push(bpf_stmt(BPF_RET | BPF_K, u32::MAX));

        let prog = sock_fprog {
            len: filters.len() as u16,
            filter: filters.as_mut_ptr() as *mut sock_filter,
        };
        let fd = group[0].socket.as_raw_fd();
        let ret = unsafe {
            setsockopt(
                fd,
                SOL_SOCKET,
                SO_ATTACH_REUSEPORT_CBPF,
                &prog as *const _ as *const c_void,
                size_of_val(&prog) as socklen_t,
            )
        };
        ensure!(ret != -1, std::io::Error::last_os_error());
        Ok(())
    }

    pub(crate) fn handle_outbound_ip_udp(&self, local_id: u8) {
        let mut buffer = [0u8; 1500];

        // Pre-initialize with our Meta header (local -> remote)
        let meta = Meta {
            src_id: local_id,
            dst_id: self.config.remote_id,
            reversed: 0,
        };
        buffer[..META_SIZE].copy_from_slice(meta.as_bytes());

        loop {
            let n = self.tun.recv(&mut buffer[META_SIZE..]).unwrap(); // recv 失败直接 panic
            let guard = pin();
            let endpoint_ref = self.endpoint.load(Ordering::Relaxed, &guard);
            if let Some(endpoint) = unsafe { endpoint_ref.as_ref() } {
                self.encrypt(&mut buffer[META_SIZE..META_SIZE + n]);
                let _ = self.socket.send_to(&buffer[..META_SIZE + n], endpoint);
            }
        }
    }

    pub(crate) fn handle_inbound_ip_udp(&self, local_secret: &[u8; 32]) {
        let mut recv_buf = [MaybeUninit::uninit(); 1500];
        loop {
            // 收到一个非法报文只丢弃一个报文
            let (len, addr) = { self.socket.recv_from(&mut recv_buf).unwrap() }; // recv 出错直接 panic
            let packet = unsafe { std::slice::from_raw_parts_mut(recv_buf.as_mut_ptr().cast(), len) };
            // if addr.is_ipv6() { println!("{:X?}", packet) }
            // 只有 ipv4 raw 会给 IP报头
            let offset = if self.config.family == Domain::IPV4 && self.config.schema == Schema::IP {
                (packet[0] & 0x0f) as usize * 4
            } else {
                0
            } + META_SIZE;

            {
                let guard = pin();
                let current_shared = self.endpoint.load(Ordering::Relaxed, &guard);
                let is_same = unsafe { current_shared.as_ref() }.map(|c| *c == addr).unwrap_or(false);
                if !is_same {
                    let new_shared = crossbeam::epoch::Owned::new(addr).into_shared(&guard);
                    let old_shared = self.endpoint.swap(new_shared, Ordering::Release, &guard);
                    unsafe { guard.defer_destroy(old_shared) }
                }
            }

            let payload = &mut packet[offset..];
            self.decrypt(payload, &local_secret);
            let _ = self.tun.send(payload);
        }
    }

    pub(crate) fn handle_outbound_tcp(&self, connection: &Socket) {
        let _ = (|| -> Result<()> {
            let mut buffer = [0u8; 1500];
            loop {
                let n = self.tun.recv(&mut buffer)?;
                self.encrypt(&mut buffer[..n]);
                Router::send_all_tcp(&connection, &buffer[..n])?;
            }
        })();
        let _ = connection.shutdown(Shutdown::Both);
    }
    pub(crate) fn handle_inbound_tcp(&self, connection: &Socket, local_secret: &[u8; 32]) {
        let _ = (|| -> Result<()> {
            let mut buf = [MaybeUninit::uninit(); 1500];
            let packet: &mut [u8] = unsafe { std::slice::from_raw_parts_mut(buf.as_mut_ptr().cast(), buf.len()) };
            loop {
                Router::recv_exact_tcp(&connection, &mut buf[0..6])?;
                self.decrypt2(packet, &local_secret, 0..6);
                let version = packet[0] >> 4;
                let total_len = match version {
                    4 => u16::from_be_bytes([packet[2], packet[3]]) as usize,
                    6 => u16::from_be_bytes([packet[4], packet[5]]) as usize + 40,
                    _ => bail!("Invalid IP version"),
                };
                ensure!(6 < total_len && total_len <= buf.len(), "Invalid total length");
                Router::recv_exact_tcp(&connection, &mut buf[6..total_len])?;
                self.decrypt2(packet, &local_secret, 6..total_len);
                self.tun.send(&packet[..total_len])?;
            }
        })();
        let _ = connection.shutdown(Shutdown::Both);
    }

    pub(crate) fn recv_exact_tcp(sock: &Socket, mut buf: &mut [MaybeUninit<u8>]) -> Result<()> {
        while !buf.is_empty() {
            let n = sock.recv(buf)?;
            ensure!(n != 0, std::io::ErrorKind::UnexpectedEof);
            buf = &mut buf[n..];
        }
        Ok(())
    }

    pub(crate) fn send_all_tcp(sock: &Socket, mut buf: &[u8]) -> Result<()> {
        while !buf.is_empty() {
            let n = sock.send(buf)?;
            buf = &buf[n..];
        }
        Ok(())
    }

    fn create_tun_device(config: &ConfigRouter) -> Result<Device> {
        let mut tun_config = tun::Configuration::default();
        tun_config.tun_name(config.dev.as_str()).up();

        let dev = tun::create(&tun_config)?;
        Ok(dev)
    }
    fn run_up_script(config: &ConfigRouter) -> Result<ExitStatus> {
        Ok(Command::new("sh").args(["-c", config.up.as_str()]).status()?)
    }

    fn create_endpoint(config: &ConfigRouter) -> Atomic<SockAddr> {
        match (config.endpoint.clone(), config.dst_port)
            .to_socket_addrs()
            .unwrap_or_default()
            .filter(|a| match config.family {
                Domain::IPV4 => a.is_ipv4(),
                Domain::IPV6 => a.is_ipv6(),
                _ => false,
            })
            .next()
        {
            None => Atomic::null(),
            Some(addr) => Atomic::new(addr.into()),
        }
    }

    pub fn new(config: ConfigRouter, local_id: u8) -> Result<Router> {
        let router = Router {
            tun: Self::create_tun_device(&config)?,
            endpoint: Self::create_endpoint(&config),
            socket: Self::create_socket(&config, local_id)?,
            tcp_listener_connection: Atomic::null(),
            config,
        };

        Self::run_up_script(&router.config)?;
        Ok(router)
    }

    fn bind_addr(config: &ConfigRouter) -> SockAddr {
        match config.family {
            Domain::IPV4 => SockAddr::from(SocketAddrV4::new(Ipv4Addr::UNSPECIFIED, config.src_port)),
            Domain::IPV6 => SockAddr::from(SocketAddrV6::new(Ipv6Addr::UNSPECIFIED, config.src_port, 0, 0)),
            _ => panic!("unsupported family"),
        }
    }
}

fn bpf_stmt(code: u32, k: u32) -> SockFilter {
    SockFilter::new(code as u16, 0, 0, k)
}

fn bpf_jump(code: u32, k: u32, jt: u8, jf: u8) -> SockFilter {
    SockFilter::new(code as u16, jt, jf, k)
}
