mod router;
mod config;

use crate::config::{Config, Schema};
use crate::router::{Meta, Router, META_SIZE};
use anyhow::{Context, Result};
use crossbeam::epoch::{pin, Owned};
use crossbeam_utils::thread;
use itertools::Itertools;
use std::net::Shutdown;
use std::sync::atomic::Ordering;
use std::sync::Arc;
use std::time::Duration;
use std::{collections::HashMap, env, mem::MaybeUninit};

fn main() -> Result<()> {
    println!("Starting");
    let config = serde_json::from_str::<Config>(env::args().nth(1).context("need param")?.as_str())?;

    let routers = &config
        .routers
        .iter()
        .cloned()
        .sorted_by_key(|r| r.remote_id)
        .map(|c| {
            let remote_id = c.remote_id;
            Router::new(c, config.local_id).map(|r| (remote_id, r))
        })
        .collect::<Result<HashMap<u8, Router>, _>>()?;

    for (_, group) in &routers
        .values()
        .filter(|r| r.config.schema == Schema::UDP && r.config.src_port != 0)
        .chunk_by(|r| r.config.src_port)
    {
        Router::attach_filter_udp(group.sorted_by_key(|r| r.config.remote_id).collect(), config.local_id)?;
    }

    println!("created tuns");
    const TCP_RECONNECT: u64 = 10;

    thread::scope(|s| {
        // IP, UDP
        for router in routers.values().filter(|&r| r.config.schema != Schema::TCP) {
            s.spawn(|_| {
                router.handle_outbound_ip_udp(config.local_id);
            });

            s.spawn(|_| {
                router.handle_inbound_ip_udp(&config.local_secret);
            });
        }

        for router in routers.values().filter(|&r| r.config.schema == Schema::TCP && r.config.dst_port != 0) {
            s.spawn(|_| {
                loop {
                    if let Ok(connection) = router.connect_tcp(config.local_id) {
                        let _ = thread::scope(|s| {
                            s.spawn(|_| router.handle_outbound_tcp(&connection));
                            s.spawn(|_| router.handle_inbound_tcp(&connection, &config.local_secret));
                        });
                    }
                    std::thread::sleep(Duration::from_secs(TCP_RECONNECT));
                }
            });
        }

        // tcp listeners
        for router in routers
            .values()
            .filter(|&r| r.config.schema == Schema::TCP && r.config.dst_port == 0)
            .unique_by(|r| r.config.src_port)
        {
            s.spawn(|s| {
                // accept 出错直接 panic
                loop {
                    let (connection, _) = router.socket.accept().unwrap();
                    s.spawn(move |_| {
                        connection.set_tcp_nodelay(true).unwrap();

                        let mut meta_bytes = [MaybeUninit::uninit(); META_SIZE];
                        Router::recv_exact_tcp(&connection, &mut meta_bytes).unwrap();
                        let meta: &Meta = Meta::from_bytes(&meta_bytes);
                        if meta.reversed == 0
                            && meta.dst_id == config.local_id
                            && let Some(router) = routers.get(&meta.src_id)
                        {
                            let connection = Arc::new(connection);

                            // tcp listener 只许一个连接，过来新连接就把前一个关掉。
                            {
                                let guard = pin();
                                let new_shared = Owned::new(connection.clone()).into_shared(&guard);
                                let old_shared = router.tcp_listener_connection.swap(new_shared, Ordering::Release, &guard);
                                unsafe {
                                    if let Some(old) = old_shared.as_ref() {
                                        let _ = old.shutdown(Shutdown::Both);
                                    }
                                    guard.defer_destroy(old_shared)
                                }
                            }

                            let _ = thread::scope(|s| {
                                s.spawn(|_| router.handle_outbound_tcp(&connection));
                                s.spawn(|_| router.handle_inbound_tcp(&connection, &config.local_secret));
                            });
                        }
                    });
                }
            });
        }
    })
    .unwrap();
    Ok(())
}
