use crate::router::SECRET_LENGTH;
use base64::prelude::BASE64_STANDARD;
use base64::Engine;
use serde::{Deserialize, Deserializer};
use socket2::Domain;

#[derive(Deserialize)]
pub struct Config {
    pub local_id: u8,
    #[serde(deserialize_with = "deserialize_secret")]
    pub local_secret: [u8; SECRET_LENGTH],
    pub routers: Vec<ConfigRouter>,
}
#[derive(Deserialize, Clone)]
pub struct ConfigRouter {
    pub remote_id: u8,
    #[serde(default)]
    pub schema: Schema,
    #[serde(default)]
    pub proto: u8,
    #[serde(default)]
    pub src_port: u16,
    #[serde(default)]
    pub dst_port: u16,
    #[serde(deserialize_with = "deserialize_domain")]
    pub family: Domain,
    pub mark: u32,
    pub endpoint: String,
    #[serde(deserialize_with = "deserialize_secret")]
    pub remote_secret: [u8; SECRET_LENGTH],
    pub dev: String,
    pub up: String,
}

#[derive(Deserialize, Default, PartialEq, Clone, Copy)]
pub enum Schema {
    #[default]
    IP,
    UDP,
    TCP,
}

fn deserialize_domain<'de, D>(d: D) -> Result<Domain, D::Error>
where
    D: Deserializer<'de>,
{
    match u8::deserialize(d)? {
        4 => Ok(Domain::IPV4),
        6 => Ok(Domain::IPV6),
        _ => Err(serde::de::Error::custom("Invalid domain")),
    }
}

fn deserialize_secret<'de, D>(d: D) -> Result<[u8; SECRET_LENGTH], D::Error>
where
    D: Deserializer<'de>,
{
    BASE64_STANDARD
        .decode(String::deserialize(d)?)
        .map_err(serde::de::Error::custom)?
        .as_slice()
        .try_into()
        .map_err(serde::de::Error::custom)
}
