#ifndef TUN_ROUTER_H
#define TUN_ROUTER_H

#include "Config.h"
#include <boost/core/noncopyable.hpp>
#include <b64/decode.h>
#include <netdb.h>
#include <linux/if.h>
#include <linux/if_tun.h>
#include <fcntl.h>
#include <sys/ioctl.h>

class Router : private boost::noncopyable {
public:
    static const int secret_length = 32;
    static std::map<std::pair<unsigned char, unsigned char>, int> raws; // (family,proto) => fd
    static std::map<int, Router *> tuns;                                // fd => router
    static std::map<int, Router *> all;                                 // id => router
    static unsigned char local_secret[secret_length];

    const ConfigRouter &config;
    unsigned char secret[secret_length];
    sockaddr_storage remote_addr{};
    int tun;
    int raw;

    explicit Router(const ConfigRouter &config) : config(config) {
        create_secret(config.remote_secret, secret);
        create_remote_addr();
        create_tun();
        create_raw();
        system(config.up.c_str());

        all[config.remote_id] = this;
        tuns[tun] = this;
    };

    static void create_secret(const std::string &secret, unsigned char *target) {
        base64::decoder decoder;
        decoder.decode(secret.c_str(), secret.length(), (char *) target);
    }

    void create_remote_addr() {
        if (config.endpoint.empty()) {
            addrinfo hints = {.ai_family = config.family};
            addrinfo *result;
            if (auto ret = getaddrinfo(config.endpoint.c_str(), nullptr, &hints, &result) != 0) {
                puts(gai_strerror(ret));
                throw;
            }
            remote_addr = *(sockaddr_storage *) result->ai_addr;
            freeaddrinfo(result);
        }
    }

    void create_tun() {
        ifreq ifr{};
        ifr.ifr_flags = IFF_TUN | IFF_NO_PI;
        strncpy(ifr.ifr_name, config.dev.c_str(), IFNAMSIZ);
        tun = open("/dev/net/tun", O_RDWR);
        if (tun < 0) {
            perror("tun init error");
            throw;
        }

        if (ioctl(tun, TUNSETIFF, &ifr) < 0) {
            perror("ioctl error");
            throw;
        }
    }

    void create_raw() {
        auto key = std::make_pair(config.family, config.proto);
        if (!raws.contains(key)) {
            if (auto result = socket(config.family, SOCK_RAW, config.proto) < 0) {
                perror("socket init error");
                throw;
            } else {
                raws[key] = result;
            }
        }
        raw = raws[key];
    }

    void encrypt(unsigned char *data, size_t length) {
        for (size_t i = 0; i < length; i++) {
            data[i] ^= local_secret[i % secret_length];
        }
    }

    void decrypt(unsigned char *data, size_t length) {
        for (size_t i = 0; i < length; i++) {
            data[i] ^= secret[i % secret_length];
        }
    }
};

#endif
