mod router;

use crate::router::{Router, RouterReader, RouterWriter, SECRET_LENGTH};
use std::collections::HashMap;
use std::env;
use std::error::Error;
use std::intrinsics::transmute;
use std::io::{Read, Write};
use std::mem::MaybeUninit;
use std::sync::{Arc, Mutex};
use crossbeam_utils::thread;
use grouping_by::GroupingBy;
use pnet::packet::ipv4::Ipv4Packet;
use socket2::Socket;

#[repr(C)]
pub struct Meta {
    pub src_id: u8,
    pub dst_id: u8,
    pub reversed: u16,
}

use serde::Deserialize;

#[derive(Deserialize)]
pub struct ConfigRouter {
    pub remote_id: u8,
    pub proto: i32,
    pub family: u8,
    pub mark: u32,
    pub endpoint: String,
    pub remote_secret: String,
    pub dev: String,
    pub up: String,
}

#[derive(Deserialize)]
pub struct Config {
    pub local_id: u8,
    pub local_secret: String,
    pub routers: Vec<ConfigRouter>,
}

fn main() -> Result<(), Box<dyn Error>> {
    let config: Config = serde_json::from_str(env::args().nth(1).ok_or("need param")?.as_str())?;
    let local_secret: [u8; SECRET_LENGTH] = Router::create_secret(config.local_secret.as_str())?;
    let mut sockets: HashMap<u16, Arc<Socket>> = HashMap::new();
    let routers: HashMap<u8, Router> = config
        .routers
        .iter()
        .map(|c| Router::new(c, &mut sockets).map(|router| (c.remote_id, router)))
        .collect::<Result<_, _>>()?;
    let (mut router_readers, router_writers): (
        HashMap<u8, RouterReader>,
        HashMap<u8, RouterWriter>,
    ) = routers
        .into_iter()
        .map(|(id, router)| {
            let (reader, writer) = router.split();
            ((id, reader), (id, writer))
        })
        .unzip();
    let router_writers3: Vec<(Arc<Socket>, HashMap<u8, RouterWriter>)> = router_writers
        .into_iter()
        .grouping_by(|(_, v)| v.key())
        .into_iter()
        .map(|(k, v)| {
            (
                Arc::clone(sockets.get_mut(&k).unwrap()),
                v.into_iter().collect(),
            )
        })
        .collect();
    println!("created tuns");

    // 获取系统的核心数量
    let num_threads = std::thread::available_parallelism()
        .map_or(1, |n| n.get()); // 默认使用1个线程，如果获取不到核心数

    thread::scope(|s| {
        // 根据核心数量调整线程数
        let readers_per_thread = (router_readers.len() as f32 / num_threads as f32).ceil() as usize;
        let writers_per_thread = (router_writers3.len() as f32 / num_threads as f32).ceil() as usize;

        // 将 router_readers 划分到多个线程
        let readers_chunks: Vec<_> = router_readers
            .into_iter()
            .collect::<Vec<_>>()
            .chunks(readers_per_thread)
            .map(|chunk| chunk.to_vec())
            .collect();

        // 将 router_writers3 划分到多个线程
        let writers_chunks: Vec<_> = router_writers3
            .into_iter()
            .collect::<Vec<_>>()
            .chunks(writers_per_thread)
            .map(|chunk| chunk.to_vec())
            .collect();

        // 启动处理 router_readers 的线程
        for chunk in readers_chunks {
            s.spawn(move |_| {
                for router in chunk {
                    let mut buffer = [0u8; 1500 - 20]; // minus typical IP header space
                    let meta_size = std::mem::size_of::<Meta>();

                    // Pre-initialize with our Meta header (local -> remote)
                    let meta = Meta {
                        src_id: config.local_id,
                        dst_id: router.1.config.remote_id,
                        reversed: 0,
                    };
                    // Turn the Meta struct into bytes
                    let meta_bytes = unsafe {
                        std::slice::from_raw_parts(&meta as *const Meta as *const u8, meta_size)
                    };
                    buffer[..meta_size].copy_from_slice(meta_bytes);

                    loop {
                        let n = router.1.tun_reader.read(&mut buffer[meta_size..]).unwrap();
                        if let Some(ref addr) = *router.1.endpoint.read().unwrap() {
                            router.1.encrypt(&mut buffer[meta_size..meta_size + n]);
                            #[cfg(target_os = "linux")]
                            let _ = router.1.socket.set_mark(router.1.config.mark);
                            let _ = router.1.socket.send_to(&buffer[..meta_size + n], addr);
                        }
                    }
                }
            });
        }

        // 启动处理 router_writers 的线程
        for chunk in writers_chunks {
            s.spawn(move |_| {
                for (socket, mut router_writers) in chunk {
                    let mut recv_buf = [MaybeUninit::uninit(); 1500];
                    loop {
                        let _ = (|| {
                            let (len, addr) = socket.recv_from(&mut recv_buf).unwrap();
                            let data: &mut [u8] = unsafe { transmute(&mut recv_buf[..len]) };

                            let packet = Ipv4Packet::new(data).ok_or("malformed packet")?;
                            let header_len = packet.get_header_length() as usize * 4;
                            let (_ip_header, rest) = data
                                .split_at_mut_checked(header_len)
                                .ok_or("malformed packet")?;
                            let (meta_bytes, payload) = rest
                                .split_at_mut_checked(std::mem::size_of::<Meta>())
                                .ok_or("malformed packet")?;
                            let meta: &Meta = unsafe { transmute(meta_bytes.as_ptr()) };
                            if meta.dst_id == config.local_id && meta.reversed == 0 {
                                let router = router_writers
                                    .get_mut(&meta.src_id)
                                    .ok_or("missing router")?;
                                *router.endpoint.write().unwrap() = Some(addr);
                                router.decrypt(payload, &local_secret);
                                router.tun_writer.write_all(payload)?;
                            }

                            Ok::<(), Box<dyn Error>>(())
                        })();
                    }
                }
            });
        }
    })
    .unwrap();
    Ok(())
}
