import parse from 'csv-parse';
import util from 'util';
import fs from 'fs';
import path from 'path';
import YAML from 'yaml';
import _ from 'lodash';
import * as child_process from 'child_process';

class InventoryBuilder {
  hosts: { [key: string]: any };
  gateways: any;
  connections: string[];

  async load(sheetName) {
    const data = await fs.promises.readFile(path.join('data', `内网互联计划 - ${sheetName}.csv`));
    // @ts-ignore
    return (await util.promisify(parse)(data, { columns: true, cast: true })).filter(h => h.id);
  }

  async main() {
    this.hosts = _.keyBy(await this.load('nextgen links'), 'name');
    this.gateways = _.mapValues(_.groupBy(await this.load('gateways'), 'name'), g => _.keyBy(g, 'isp'));
    this.connections = _.intersection(Object.keys(this.hosts), Object.keys(_.find(this.hosts)));
    for (const host of Object.values(this.hosts)) {
      host.wgPublickey = await this.wgPublickey(host.wgPrivateKey);
    }
    console.log(Object.values(this.hosts));
    const hosts = Object.fromEntries(Object.values(this.hosts).map(h => [h.host, this.host_vars(h)]));
    console.log(hosts);
    const result = YAML.stringify({ wg: { hosts } });
    return fs.promises.writeFile('result/inventory.yaml', result);
  }

  host_vars(host) {
    const connections = [];
    for (const h of this.connections) {
      if (h != host.name) {
        const to = host[h];
        const from = this.hosts[h][host.name];
        if (from && to) {
          // 非对称连接
          connections.push(this.parse_connection(host, this.hosts[h], to, false, host.id > this.hosts[h].id));
          connections.push(this.parse_connection(host, this.hosts[h], from, false, host.id < this.hosts[h].id));
        } else if (from || to) {
          // 对称连接
          connections.push(this.parse_connection(host, this.hosts[h], from || to, !!to));
        }
        // 不连接
      }
    }

    return {
      ansible_ssh_user: host.user,
      address: host.address,
      key: host.wgPrivateKey,
      gateways: _.mapValues(this.gateways[host.name], gw => _.pick(gw, ['mark_gateway'])),
      connections
    };
  }

  parse_connection(local: any, remote: any, connstr: string, mirror = false, inbound = false) {
    const [_metric, protocol, _params] = connstr.split(',');
    const metric = parseInt(_metric);
    const params = Object.fromEntries(new URLSearchParams(_params).entries());
    const name = inbound ? `${remote.name}-in` : remote.name;
    const localGatewayName = (mirror ? params.rif : params.lif) || params.if;
    const localGateway = localGatewayName ? this.gateways[local.name][localGatewayName] : undefined;
    const localGatewayMark = localGatewayName ? localGateway.mark : undefined;
    const remoteGatewayName = (mirror ? params.lif : params.rif) || params.if;
    const remoteGateway = remoteGatewayName ? this.gateways[remote.name][remoteGatewayName] : _.find(this.gateways[remote.name]);
    const remoteAddress = remoteGateway.address;
    const remoteMark = remote.mark;
    const localPort = inbound ? remote.port2 : remote.port;
    const remotePort = inbound ? local.port2 : local.port;
    const wgPublicKey = remote.wgPublickey;
    const localPeerAddress = inbound ? local.peerAddress2 : local.peerAddress;
    const remotePeerAddress = inbound ? remote.peerAddress2 : remote.peerAddress;
    return {
      name,
      metric,
      protocol,
      params,
      localGatewayMark,
      remoteMark,
      remoteAddress,
      localPort,
      remotePort,
      wgPublicKey,
      localPeerAddress,
      remotePeerAddress,
      inbound
    };
  }

  async wgPublickey(privateKey) {
    return new Promise((resolve, reject) => {
      const child = child_process.execFile('wg', ['pubkey'], { encoding: 'utf8' }, (error, stdout, stderr) => {
        if (stderr) {
          console.warn(stderr);
        }
        if (error) {
          reject(error);
        } else {
          resolve(stdout.trimEnd());
        }
      });
      child.stdin.end(privateKey);
    });
  }
}

new InventoryBuilder().main();

// export interface Host {
//
// }

// export interface Interface {
//
// }
