import parse from 'csv-parse';
import util from 'util';
import fs from 'fs';
import path from 'path';
import YAML from 'yaml';
import _ from 'lodash';
import child_process from 'child_process';
import assert from 'assert';
import ip from "ip";

class InventoryBuilder {
  hosts: { [key: string]: any };
  gateways: any;
  connections: string[];
  routeLists: any;

  async load(sheetName) {
    const data = await fs.promises.readFile(path.join('data', `内网互联计划 - ${sheetName}.csv`));
    // @ts-ignore
    return (await util.promisify(parse)(data, { columns: true, cast: true })).filter(h => h.id);
  }

  async loadGateways() {
    const gateways = await this.load('gateways2');
    for (let gateway of gateways) {
      gateway.selectionMark = gateway.mark + 50;
    }
    return gateways;
  }

  async main() {
    this.hosts = _.keyBy(await this.load('nextgen2'), 'name');
    this.gateways = _.mapValues(_.groupBy(await this.loadGateways(), 'router'), g => _.keyBy(g, 'isp'));
    //console.log(this.gateways);
    this.connections = _.intersection(Object.keys(this.hosts), Object.keys(_.find(this.hosts)));

    for (const host of Object.values(this.hosts)) {
      host.wgPublickey = await this.wgPublickey(host.wgPrivateKey);
    }
    // console.log(Object.values(this.hosts));
    const rawHosts = Object.values(this.hosts).map(h => [h.name, this.host_vars(h)]);
    const hosts = Object.fromEntries(rawHosts);
    // console.log(hosts);
    const vars = await this.loadUtilities();
    const rawHostsForSwitch = rawHosts.map(rh => {
      const hostVars = JSON.parse(JSON.stringify(rh[1]));
      hostVars.ansible_ssh_host = hostVars.address;
      return [rh[0], hostVars];
    });
    const switchHosts = Object.fromEntries(rawHostsForSwitch);
    const result = YAML.stringify({ wg: { hosts, vars }, switch: { hosts: switchHosts, vars: JSON.parse(JSON.stringify(vars)) } });
    return fs.promises.writeFile('result/inventory.yaml', result);
  }

  async loadUtilities() {
    const raw_utility = _.keyBy(await this.load('configurations'), 'key');
    this.routeLists = YAML.parse(fs.readFileSync(path.join('lists', 'result.yaml'), "utf8"));
    // 所有内网网段
    this.routeLists.mycard = ["10.198.0.0/16", "10.200.0.0/15", "172.16.0.0/12"];
    for (const h in this.hosts) {
      const host = this.hosts[h]
      for (const c of host.subnets.split(",")) {
        if (!c.length) {
          continue;
        }
        this.routeLists.mycard.push(c);
      }
    }
    // temp user before gateways
    this.routeLists.ladder_needed = raw_utility.ladderNeeded.value.split(",").map((m) => { return m.trim() });
    const vars = {
      routeLists: this.routeLists,
      routeListNames: Object.keys(this.routeLists),
    };
    for (let col in raw_utility) {
      vars[col] = raw_utility[col].value;
    }

    return vars;
  }
  host_vars(host) {
    const connections = [];
    host.dockerServices = {
      version: '2.4',
      services: {
        bird: {
          restart: "always",
          image: "git-registry.mycard.moe/nanahira/docker-bird",
          network_mode: "host",
          cap_add: ["NET_ADMIN", "NET_BROADCAST", "NET_RAW"],
          volumes: ["./bird.conf:/etc/bird/bird.conf:ro"]
        }
      }
    };
    host.frpsNeeded = false;
    const null_connection = "10000,null";
    const lanInterfaces = host.lanInterfaces.length > 0 ? host.lanInterfaces.split(",") : [];
    const routePlans = [];
    for (const h of this.connections) {
      if (h != host.name) {
        const to = host[h]; // 当前主机的条目
        const from = this.hosts[h][host.name]; // 其他主机的这个主机的条目
        if (from && to) {
          // 非对称连接
          connections.push(this.parse_connection(host, this.hosts[h], to, false, true, false));
          connections.push(this.parse_connection(host, this.hosts[h], from, true, false, true));
        } else if (from || to) {
          // 对称连接
          const connectionString = from || to;
          connections.push(this.parse_connection(host, this.hosts[h], connectionString, true, true, connectionString === from));
          connections.push(this.parse_connection(host, this.hosts[h], null_connection, false, false, false));
        } else {
          // 不连接
          connections.push(this.parse_connection(host, this.hosts[h], null_connection, true, false, false));
          connections.push(this.parse_connection(host, this.hosts[h], null_connection, false, true, false));
        }
        routePlans.push({
          name: h.replace(/-/g, "_"),
          destMark: this.hosts[h].destMark,
          address: this.hosts[h].address
        });
      }
    }

    return {
      ansible_ssh_host: host.host,
      ansible_ssh_user: host.user,
      ansible_ssh_port: host.sshPort || 22,
      ansible_python_interpreter: host.python || "python3",
      address: host.address,
      isCN: host.location.startsWith("CN"),
      key: host.wgPrivateKey,
      frpsNeeded: host.frpsNeeded,
      frpsPort: host.frpsPort,
      gateways: _.values(this.gateways[host.name]),
      connections,
      lanInterfaces,
      dockerServices: host.dockerServices,
      routePlans
    };
  }

  parse_connection(local: any, remote: any, connstr: string, inbound: boolean, outbound: boolean, reverse: boolean) {
    const leftbottom = local.id > remote.id; // true 条目位于左下，false 条目位于右上
    const cis = !reverse; // true 无需翻转，false 需要翻转。
    const primary = leftbottom ? outbound : inbound; // true 使用 peerAddress、port, false 使用peerAddress2、port2
    const connStrSplited = connstr.split(',');
    const [_metric, protocol] = connStrSplited;
    const paramsString = connStrSplited.slice(2).join("&");
    const metric = parseInt(_metric);
    const params = Object.fromEntries(new URLSearchParams(paramsString).entries());
    const name = `mc${!outbound ? 'i' : '-'}${remote.name}`;
    const localGatewayName = (cis ? params.lif : params.rif) || params.if;
    const localGateway = localGatewayName ? this.gateways[local.name][localGatewayName] : _.find(this.gateways[local.name]);
    //console.log(local.name, paramsString, params, localGatewayName, localGateway.name)
    const localGatewayMark = parseInt(localGateway.selectionMark);
    const remoteGatewayName = (cis ? params.rif : params.lif) || params.if;
    const remoteGateway = remoteGatewayName ? this.gateways[remote.name][remoteGatewayName] : _.find(this.gateways[remote.name]);
    //const remoteGatewayMark = remoteGatewayName ? remoteGateway.mark : undefined;
    //console.log(remoteGateway.name);
    const remoteAddress = remoteGateway.address;
    const remoteLocalAddress = remote.address;
    const remoteNextMark = remote.nextMark;
    const remoteDestMark = remote.destMark;
    const localPort = (primary ? remote.port : remote.port2) + local.offset;
    const remotePort = (primary ? local.port : local.port2) + remote.offset;
    const remoteFrpsPort = remote.frpsPort;
    const wgPublicKey = remote.wgPublickey;
    const localPeerAddress = primary ? `10.200.${local.id}.${remote.id}` : `10.201.${local.id}.${remote.id}`;
    const remotePeerAddress = primary ? `10.200.${remote.id}.${local.id}` : `10.201.${remote.id}.${local.id}`;
    const link6Address = `fe80::${primary ? 1 : 2}:${local.id}:${remote.id}/64`;

    const frpType = protocol === 'wgfrp' ? (this.gatewayCompare(localGateway, remoteGateway) ? 'frps' : 'frpc') : undefined;

    if (frpType === "frps" && !local.dockerServices.services.frps) {
      local.frpsNeeded = true;
      local.dockerServices.services.frps = {
        restart: "always",
        image: "fatedier/frps:v0.34.2",
        network_mode: "host",
        command: "-c /frps.ini",
        volumes: ["./frps.ini:/frps.ini:ro"]
      }
    }

    if (frpType === "frpc") {
      local.dockerServices.services[`frpc-${name}`] = {
        restart: "always",
        image: "fatedier/frpc:v0.34.2",
        network_mode: "host",
        command: "-c /frpc.ini",
        volumes: [`./frpc-${name}.ini:/frpc.ini:ro`]
      }
    }

    const mtu = Math.min(localGateway ? localGateway.mtu : 1500, remoteGateway ? remoteGateway.mtu : 1500);
    //console.log(local.name, name, mtu);

    if (outbound) {
      console.log(`${local.name} GW ${localGateway.isp} ${inbound ? "<" : "="}==[${protocol}]==> ${remote.name} GW ${remoteGateway.isp}`);
    }

    return {
      name,
      metric,
      protocol,
      params,
      localGatewayMark,
      remoteNextMark,
      remoteDestMark,
      remoteAddress,
      remoteLocalAddress,
      localPort,
      remotePort,
      wgPublicKey,
      localPeerAddress,
      remotePeerAddress,
      link6Address,
      remoteFrpsPort,
      frpType,
      inbound,
      outbound,
      mtu
    };
  }

  // frps还是frpc的积分，NAT越有利分越高
  gatewayCompareScore(gateway: any): number {
    let score: number = 0xff - gateway.id;
    const ipv4Score = ({
      "static": 2,
      "dynamic": 1
    })[gateway.ipv4] || 0;
    score |= ipv4Score << 12;
    const ipv4NatScore = ({
      "ports": 0,
      "dmz": 1
    })[gateway.ipv4Nat] || 2;
    score |= ipv4NatScore << 10;
    const globalSSHScore = ({
      "globalssh": 1
    })[gateway.ssh] || 0;
    score |= globalSSHScore << 13;
    return score;
  }

  // true: 本地做 server，false: 远端做server
  // 如果都不能做，抛异常
  // 两个参数对调返回的结果必须相反
  gatewayCompare(localGateway: any, remoteGateway: any): boolean {
    // 两边至少一个有IPv4地址才能连
    assert(localGateway.ipv4 !== '' || remoteGateway.ipv4 !== '');
    // 两边必须相反
    assert(this.gatewayCompareScore(localGateway) !== this.gatewayCompareScore(remoteGateway));
    // 两边只有一边是GlobalSSH的，GlobalSSH做s
    // 只有一边有static的，就static做s
    // 如果都static，那么没有NAT的或者dmz的做s
    // 如果都还相同，就让列表中更靠前的做s
    return this.gatewayCompareScore(localGateway) > this.gatewayCompareScore(remoteGateway);
  }

  async wgPublickey(privateKey) {
    return new Promise((resolve, reject) => {
      const child = child_process.execFile('wg', ['pubkey'], { encoding: 'utf8' }, (error, stdout, stderr) => {
        if (stderr) {
          console.warn(stderr);
        }
        if (error) {
          reject(error);
        } else {
          resolve(stdout.trimEnd());
        }
      });
      child.stdin.end(privateKey);
    });
  }
}

new InventoryBuilder().main();

// export interface Host {
//
// }

// export interface Interface {
//
// }
