mod config;
mod router;

use crate::config::{Config, Schema};
use crate::router::{META_SIZE, Meta, Router};
use anyhow::{Context, Result};
use crossbeam::epoch::{Owned, pin};
use crossbeam_utils::thread;
use itertools::Itertools;
use std::net::Shutdown;
use std::sync::Arc;
use std::sync::atomic::Ordering;
use std::time::Duration;
use std::{collections::HashMap, env, mem::MaybeUninit};

fn main() -> Result<()> {
    println!("Starting");
    let config = serde_json::from_str::<Config>(env::args().nth(1).context("need param")?.as_str())?;

    let routers = &config
        .routers
        .into_iter()
        .sorted_by_key(|r| r.remote_id)
        .map(|c| Router::new(c).map(|router| (router.config.remote_id, router)))
        .collect::<Result<HashMap<u8, Router>, _>>()?;

    for (_, group) in &routers
        .values()
        .filter(|r| r.config.schema == Schema::UDP && r.config.src_port != 0)
        .chunk_by(|r| r.config.src_port)
    {
        Router::attach_filter_udp(group.sorted_by_key(|r| r.config.remote_id).collect())?;
    }

    println!("created tuns");
    const TCP_RECONNECT: u64 = 10;

    thread::scope(|s| {
        // IP, UDP
        for router in routers.values().filter(|&r| r.config.schema != Schema::TCP) {
            s.spawn(|_| {
                router.handle_outbound_ip_udp();
            });

            s.spawn(|_| {
                router.handle_inbound_ip_udp();
            });
        }

        for router in routers.values().filter(|&r| r.config.schema == Schema::TCP && r.config.dst_port != 0) {
            s.spawn(|_| {
                loop {
                    if let Ok(connection) = router.connect_tcp() {
                        let _ = thread::scope(|s| {
                            s.spawn(|_| router.handle_outbound_tcp(&connection));
                            s.spawn(|_| router.handle_inbound_tcp(&connection));
                        });
                    }
                    std::thread::sleep(Duration::from_secs(TCP_RECONNECT));
                }
            });
        }

        // tcp listeners
        for router in routers
            .values()
            .filter(|&r| r.config.schema == Schema::TCP && r.config.dst_port == 0)
            .unique_by(|r| r.config.src_port)
        {
            s.spawn(|s| {
                // listen 或 accept 出错直接 panic
                router.listen_tcp().unwrap();
                loop {
                    let (connection, _) = router.socket.accept().unwrap();
                    s.spawn(move |_| {
                        connection.set_tcp_nodelay(true).unwrap();

                        let mut meta_bytes = [MaybeUninit::uninit(); META_SIZE];
                        Router::recv_exact_tcp(&connection, &mut meta_bytes).unwrap();
                        let meta: &Meta = Meta::from_bytes(&meta_bytes);
                        if meta.reversed == 0
                            && let Some(router) = routers.get(&meta.src_id)
                            && meta.dst_id == router.config.local_id
                        {
                            let connection = Arc::new(connection);

                            // tcp listener 只许一个连接，过来新连接就把前一个关掉。
                            {
                                let guard = pin();
                                let new_shared = Owned::new(connection.clone()).into_shared(&guard);
                                let old_shared = router.tcp_listener_connection.swap(new_shared, Ordering::Release, &guard);
                                unsafe {
                                    if let Some(old) = old_shared.as_ref() {
                                        let _ = old.shutdown(Shutdown::Both);
                                    }
                                    guard.defer_destroy(old_shared)
                                }
                            }

                            let _ = thread::scope(|s| {
                                s.spawn(|_| router.handle_outbound_tcp(&connection));
                                s.spawn(|_| router.handle_inbound_tcp(&connection));
                            });
                        }
                    });
                }
            });
        }
    })
    .unwrap();
    Ok(())
}
