use socket2::{Domain, Protocol, SockAddr, Socket, Type};
use std::collections::hash_map::Entry;
use std::collections::HashMap;
use std::net::ToSocketAddrs;
use std::process::{Command, ExitStatus};
use std::sync::{Arc, RwLock};
use tun::{Reader, Writer};
pub const SECRET_LENGTH: usize = 32;
use crate::ConfigRouter;
use base64::prelude::*;


#[inline]
fn xor_with_secret(data: &mut [u8], secret: &[u8; SECRET_LENGTH]) {
    let mut i = 0;
    let len = data.len();

    // 处理完整块
    while i + SECRET_LENGTH <= len {
        for j in 0..SECRET_LENGTH {
            data[i + j] ^= secret[j];
        }
        i += SECRET_LENGTH;
    }

    // 处理剩余尾部
    for j in 0..len - i {
        data[i + j] ^= secret[j];
    }
}

// tun -> raw
pub struct RouterReader<'a> {
    pub config: &'a ConfigRouter,
    pub secret: [u8; SECRET_LENGTH],
    pub tun_reader: Reader,
    pub socket: Arc<Socket>,
    pub endpoint: Arc<RwLock<Option<SockAddr>>>,
}

impl<'a> RouterReader<'a> {
    #[inline]
    pub(crate) fn encrypt(&self, data: &mut [u8]) {
        xor_with_secret(data, &self.secret);
    }
}

// raw -> tun
pub struct RouterWriter<'a> {
    pub config: &'a ConfigRouter,
    pub secret: [u8; SECRET_LENGTH],
    pub tun_writer: Writer,
    pub endpoint: Arc<RwLock<Option<SockAddr>>>,
}

impl<'a> RouterWriter<'a> {
    #[inline]
    pub(crate) fn decrypt(&self, data: &mut [u8]) {
        xor_with_secret(data, &self.secret);
    }

    pub(crate) fn key(&self) -> u16 {
        Router::key(self.config)
    }
}

pub struct Router<'a> {
    pub config: &'a ConfigRouter,
    pub local_secret: [u8; SECRET_LENGTH],
    pub remote_secret: [u8; SECRET_LENGTH],
    pub tun_reader: Reader,
    pub tun_writer: Writer,
    pub socket: Arc<Socket>,
    pub endpoint: Arc<RwLock<Option<SockAddr>>>,
}

impl<'a> Router<'a> {
    pub(crate) fn create_secret(
        config: &str,
    ) -> Result<[u8; SECRET_LENGTH], Box<dyn std::error::Error>> {
        let mut secret = [0u8; SECRET_LENGTH];
        let decoded = BASE64_STANDARD.decode(config)?;
        let len = decoded.len().min(SECRET_LENGTH);
        secret[..len].copy_from_slice(&decoded[..len]);
        Ok(secret)
    }

    fn key(config: &ConfigRouter) -> u16 {
        (config.family as u16) << 8 | config.proto as u16
    }
    fn create_raw_socket(
        config: &ConfigRouter,
        sockets: &mut HashMap<u16, Arc<Socket>>,
    ) -> Result<Arc<Socket>, Box<dyn std::error::Error>> {
        let key = Router::key(config);
        let result = match sockets.entry(key) {
            Entry::Occupied(entry) => entry.get().clone(),
            Entry::Vacant(entry) => entry
                .insert(Arc::new(Socket::new(
                    if config.family == 6 {
                        Domain::IPV6
                    } else {
                        Domain::IPV4
                    },
                    Type::RAW,
                    Some(Protocol::from(config.proto)),
                )?))
                .clone(),
        };
        Ok(result)
    }
    fn create_tun_device(
        config: &ConfigRouter,
    ) -> Result<(Reader, Writer), Box<dyn std::error::Error>> {
        let mut tun_config = tun::Configuration::default();
        tun_config.tun_name(config.dev.as_str()).up();

        let dev = tun::create(&tun_config)?;
        Ok(dev.split())
    }
    fn run_up_script(config: &ConfigRouter) -> std::io::Result<ExitStatus> {
        Command::new("sh").args(["-c", config.up.as_str()]).status()
    }

    fn create_endpoint(
        config: &ConfigRouter,
    ) -> Result<Arc<RwLock<Option<SockAddr>>>, Box<dyn std::error::Error>> {
        let parsed = (config.endpoint.clone(), 0u16)
            .to_socket_addrs()?
            .next()
            .ok_or(config.endpoint.clone())?;
        Ok(Arc::new(RwLock::new(Some(parsed.into()))))
    }

    pub fn new(
        config: &'a ConfigRouter,
        sockets: &mut HashMap<u16, Arc<Socket>>,
    ) -> Result<Router<'a>, Box<dyn std::error::Error>> {
        let local_secret = Self::create_secret(config.local_secret.as_str())?;
        let remote_secret = Self::create_secret(config.remote_secret.as_str())?;
        let endpoint = Self::create_endpoint(&config)?;
        let socket = Self::create_raw_socket(&config, sockets)?;
        if (config.mark > 0) {
            #[cfg(target_os = "linux")]
            let _ = socket.set_mark(config.mark);
        }
        let (tun_reader, tun_writer) = Self::create_tun_device(&config)?;
        Self::run_up_script(&config)?;

        let router = Router {
            config,
            local_secret,
            remote_secret,
            endpoint,
            tun_reader,
            tun_writer,
            socket,
        };

        Ok(router)
    }

    pub fn split(self) -> (RouterReader<'a>, RouterWriter<'a>) {
        let writer = RouterWriter {
            config: self.config,
            secret: self.local_secret,
            endpoint: Arc::clone(&self.endpoint),
            tun_writer: self.tun_writer,
        };

        let reader = RouterReader {
            config: self.config,
            secret: self.remote_secret,
            endpoint: self.endpoint,
            tun_reader: self.tun_reader,
            socket: self.socket,
        };

        (reader, writer)
    }
}
