import parse from 'csv-parse';
import util from 'util';
import fs from 'fs';
import path from 'path';
import YAML from 'yaml';
import _ from 'lodash';
import * as child_process from 'child_process';

class InventoryBuilder {
  hosts: { [key: string]: any };
  gateways: any;
  connections: string[];

  async load(sheetName) {
    const data = await fs.promises.readFile(path.join('data', `内网互联计划 - ${sheetName}.csv`));
    // @ts-ignore
    return (await util.promisify(parse)(data, { columns: true, cast: true })).filter(h => h.id);
  }

  async main() {
    this.hosts = _.keyBy(await this.load('nextgen links'), 'name');
    this.gateways = _.mapValues(_.groupBy(await this.load('gateways'), 'name'), g => _.keyBy(g, 'isp'));
    this.connections = _.intersection(Object.keys(this.hosts), Object.keys(_.find(this.hosts)));
    for (const host of Object.values(this.hosts)) {
      host.wgPublickey = await this.wgPublickey(host.wgPrivateKey);
    }
    // console.log(Object.values(this.hosts));
    const hosts = Object.fromEntries(Object.values(this.hosts).map(h => [h.host, this.host_vars(h)]));
    // console.log(hosts);
    const result = YAML.stringify({ wg: { hosts } });
    return fs.promises.writeFile('result/inventory.yaml', result);
  }

  host_vars(host) {
    const connections = [];
    for (const h of this.connections) {
      if (h != host.name) {
        const to = host[h];
        const from = this.hosts[h][host.name];
        if (from && to) {
          // 非对称连接
          connections.push(this.parse_connection(host, this.hosts[h], to, false, true));
          connections.push(this.parse_connection(host, this.hosts[h], from, true, false));
        } else if (from || to) {
          // 对称连接
          connections.push(this.parse_connection(host, this.hosts[h], from || to, true, true));
        }
        // 不连接
      }
    }

    return {
      ansible_ssh_user: host.user,
      address: host.address,
      key: host.wgPrivateKey,
      gateways: _.mapValues(this.gateways[host.name], gw => _.pick(gw, ['mark_gateway'])),
      connections
    };
  }

  parse_connection(local: any, remote: any, connstr: string, inbound: boolean, outbound: boolean) {
    const leftbottom = local.id > remote.id; // true 条目位于左下，false 条目位于右上
    const cis = !(!leftbottom && inbound && outbound); // true 无需翻转，false 需要翻转。
    const primary = leftbottom ? outbound : inbound; // true 使用 peerAddress、port, false 使用peerAddress2、port2

    const [_metric, protocol, _params] = connstr.split(',');
    const metric = parseInt(_metric);
    const params = Object.fromEntries(new URLSearchParams(_params).entries());
    const name = `mc${!outbound ? 'i' : '-'}${remote.name}`;
    const localGatewayName = (cis ? params.lif : params.rif) || params.if;
    const localGateway = localGatewayName ? this.gateways[local.name][localGatewayName] : undefined;
    const localGatewayMark = localGatewayName ? localGateway.mark : undefined;
    const remoteGatewayName = (cis ? params.rif : params.lif) || params.if;
    const remoteGateway = remoteGatewayName ? this.gateways[remote.name][remoteGatewayName] : _.find(this.gateways[remote.name]);
    const remoteAddress = remoteGateway.address;
    const remoteMark = remote.mark;
    const localPort = primary ? remote.port : remote.port2;
    const remotePort = primary ? local.port : local.port2;
    const wgPublicKey = remote.wgPublickey;
    const localPeerAddress = primary ? `10.200.${local.id}.${remote.id}` : `10.201.${local.id}.${remote.id}`;
    const remotePeerAddress = primary ? `10.200.${remote.id}.${local.id}` : `10.201.${remote.id}.${local.id}`;
    return {
      name,
      metric,
      protocol,
      params,
      localGatewayMark,
      remoteMark,
      remoteAddress,
      localPort,
      remotePort,
      wgPublicKey,
      localPeerAddress,
      remotePeerAddress,
      inbound,
      outbound
    };
  }

  async wgPublickey(privateKey) {
    return new Promise((resolve, reject) => {
      const child = child_process.execFile('wg', ['pubkey'], { encoding: 'utf8' }, (error, stdout, stderr) => {
        if (stderr) {
          console.warn(stderr);
        }
        if (error) {
          reject(error);
        } else {
          resolve(stdout.trimEnd());
        }
      });
      child.stdin.end(privateKey);
    });
  }
}

new InventoryBuilder().main();

// export interface Host {
//
// }

// export interface Interface {
//
// }
