use crate::shared::data::{self, DATABASE, GatewayGroup, GatewayGroupID, GatewayID, RegionID, RouterID};
use std::collections::{BTreeMap, BTreeSet};
use std::sync::LazyLock;

pub static GATEWAYGROUPINDEX: LazyLock<BTreeMap<GatewayGroupID, Vec<&'static data::Gateway>>> = LazyLock::new(|| {
    DATABASE
        .gateway_groups
        .iter()
        .map(|g| {
            let routers = g.search_routers(&DATABASE.routers, &DATABASE.gateway_groups);
            (g.id, DATABASE.gateways.iter().filter(|gw| routers.contains(&gw.router)).collect())
        })
        .collect()
});

impl GatewayGroup {
    fn search_routers(&self, routers_data: &[data::Router], groups_data: &[Self]) -> BTreeSet<RouterID> {
        let mut routers: BTreeSet<RouterID> = self
            .children
            .iter()
            .flat_map(|c| groups_data.iter().find(|g| &g.name == c))
            .flat_map(|g| g.search_routers(routers_data, groups_data))
            .chain(routers_data.iter().filter(|r| self.location_prefix.iter().any(|p| r.location.starts_with(p))).map(|r| r.id))
            .chain(self.include_routers.iter().cloned())
            .collect();
        for r in &self.exclude_routers {
            routers.remove(r);
        }
        routers
    }

    pub fn default_via(id: RouterID) -> BTreeMap<RouterID, RouterID> {
        DATABASE.routers.iter().filter(|r| r.id != id).map(|r| (r.id, r.id)).collect()
    }

    pub fn default_plan(id: RouterID) -> BTreeMap<RegionID, BTreeMap<GatewayGroupID, GatewayID>> {
        DATABASE
            .regions
            .iter()
            .enumerate()
            .map(|(r, _)| {
                (
                    RegionID(r as u8),
                    GATEWAYGROUPINDEX.iter().map(|(&gid, g)| (gid, g.iter().min_by_key(|gw| Self::guess(id, gw, r)).unwrap().id)).collect(),
                )
            })
            .collect()
    }

    pub fn apply(
        self_via: &mut BTreeMap<RouterID, RouterID>,
        new_via: &mut BTreeMap<RouterID, RouterID>,
        self_plan: &mut BTreeMap<RegionID, BTreeMap<GatewayGroupID, GatewayID>>,
        new_plan: &mut BTreeMap<RegionID, BTreeMap<GatewayGroupID, GatewayID>>,
    ) {
        self_via.append(new_via);
        for (region, inner) in new_plan {
            if let Some(p) = self_plan.get_mut(region) {
                p.append(inner);
            }
        }
    }

    fn guess(id: RouterID, gw: &data::Gateway, region: usize) -> i32 {
        gw.metrics[region].saturating_add(gw.cost_outbound).saturating_add(if gw.router == id { 0 } else { 100 })
    }
}
