use crate::quality::Quality;
use crate::settings::{HALF_LIFE, PENALTY, PENALTY_MIN};
use crate::shared::{
    data::{self, DATABASE, GatewayGroupID, GatewayID, RegionID, RouterID},
    protocol::{Downlink, MessageType, PeerQuality, Uplink},
};
use crate::{UpdatingState, shared};
use serde::Serialize;
use std::collections::BTreeMap;
use std::i32;
use std::net::SocketAddr;
use tokio::time::Instant;

#[derive(Serialize, Clone)]
pub struct Router {
    pub id: RouterID,
    #[serde(skip)]
    pub version: u32,
    pub peers: BTreeMap<RouterID, PeerQuality>,                        // quality from peer to self. HashMap 的 key 是 from.
    pub via: BTreeMap<RouterID, RouterID>,                             // dst router_id -> next hop router_id
    pub plan: BTreeMap<RegionID, BTreeMap<GatewayGroupID, GatewayID>>, // group id -> region id -> gateway_id
    #[serde(skip)]
    pub last_seen: Instant,
    #[serde(skip)]
    pub last_update: Instant,
    #[serde(skip)]
    pub addr: Option<SocketAddr>,
}

impl PartialEq<Self> for Router {
    fn eq(&self, other: &Self) -> bool {
        self.id == other.id
    }
}

impl Eq for Router {}

impl Router {
    pub fn new(id: RouterID) -> Self {
        Self {
            id,
            version: rand::random(),
            peers: DATABASE
                .connections
                .iter()
                .filter(|(_, to)| to.contains_key(&id))
                .map(|(&from, _)| (from, Default::default()))
                .collect(),
            via: shared::data::GatewayGroup::default_via(id),
            plan: shared::data::GatewayGroup::default_plan(id),
            last_seen: Instant::now(),
            last_update: Instant::now(),
            addr: None,
        }
    }

    pub fn online(&mut self, addr: SocketAddr, now: Instant) {
        if self.addr.is_none() {
            tracing::info!("router {:?} online", self.id);
        }
        self.addr = Some(addr);
        self.last_seen = now;
    }

    pub fn offline(&mut self) {
        if self.addr.is_some() {
            tracing::info!("router {:?} offline", self.id);
        }
        self.addr = None;
    }

    pub fn is_online(&self) -> bool {
        self.addr.is_some()
    }

    pub fn on_message(&mut self, uplink: &mut Uplink, addr: SocketAddr, updating: &mut UpdatingState, now: Instant) -> Option<Downlink> {
        if uplink.peers.len() == self.peers.len() {
            for (current, new) in self.peers.values_mut().zip(&mut uplink.peers) {
                *current = *new
            }
        } else if uplink.peers.len() != 0 {
            tracing::error!("router {:?} peers count wrong. local {} remote {}", self.id, self.peers.len(), uplink.peers.len());
        }

        match uplink.action {
            MessageType::Query => Some(Downlink {
                action: MessageType::Full,
                version: self.version,
                ack: uplink.version,
                via: self.via.iter().filter(|(k, v)| k != v).map(|(&k, &v)| (k, v)).collect(),
                plan: self.plan.clone(),
            }),
            MessageType::Full => {
                if uplink.version == self.version {
                    for (to, via) in self.via.iter_mut() {
                        *via = *to;
                    }
                    data::GatewayGroup::apply(&mut self.via, &mut uplink.via, &mut self.plan, &mut uplink.plan);
                    self.online(addr, now);
                }
                None
            }
            MessageType::Update => {
                if uplink.version == self.version {
                    self.online(addr, now);
                    if updating.router_id == self.id {
                        updating.router_id = Default::default();
                        data::GatewayGroup::apply(&mut self.via, &mut updating.message.via, &mut self.plan, &mut updating.message.plan);
                        self.last_update = now;
                    }
                    None
                } else if !self.is_online() {
                    Some(Downlink {
                        action: MessageType::Query,
                        version: self.version,
                        ack: uplink.version,
                        via: Default::default(),
                        plan: Default::default(),
                    })
                } else if updating.router_id == self.id && uplink.version == self.version - 1 {
                    Some(updating.message.clone()) // 重传
                } else {
                    tracing::info!("router {:?} wrong ack ack={}, seq={}", self.id, uplink.version, self.version);
                    None
                }
            }
        }
    }

    pub fn update(&self, now: Instant, routers: &BTreeMap<RouterID, Router>) -> Option<Downlink> {
        let penalty = PENALTY_MIN + (PENALTY as f32 * f32::exp2(-now.duration_since(self.last_update).div_duration_f32(HALF_LIFE))) as i32;
        let mut changed_via = BTreeMap::new();
        let mut changed_plan = BTreeMap::new();
        let mut metrics = BTreeMap::new();
        metrics.insert(self.id, 0);
        let mut overcome = false;

        // Route updates
        for to in routers.values().filter(|&r| r != self) {
            let current_via = &routers[&self.via[&to.id]];
            let current_metric = self.route_metric(to, current_via, routers);
            let (best_via, best_metric) = match DATABASE.connections[&self.id]
                .keys()
                .map(|id| &routers[id])
                .map(|r| (r, self.route_metric(to, r, routers)))
                .min_by_key(|(_, m)| *m)
                .unwrap()
            {
                (_, i32::MAX) => (to.id, i32::MAX),
                (r, m) => (r.id, m),
            };

            metrics.insert(to.id, best_metric);

            if current_via.id != best_via {
                changed_via.insert(to.id, best_via);

                if best_metric == i32::MAX || best_metric.saturating_add(penalty) < current_metric {
                    overcome = true;
                }
            }
        }

        // Plan updates (Gateways)
        for (region_id, plan) in &self.plan {
            for (&gid, gateways) in crate::shared::gateway_group::GATEWAYGROUPINDEX.iter() {
                let current_gw = plan[&gid];
                let current_metric = metrics[&DATABASE.gateways.iter().find(|f| f.id == current_gw).unwrap().router];

                let (best_gw, best_metric) = gateways
                    .iter()
                    .map(|g| (g, metrics[&g.router].saturating_add(g.cost_outbound).saturating_add(g.metrics[region_id.0 as usize])))
                    .min_by_key(|(_, m)| *m)
                    .unwrap();

                if current_gw != best_gw.id {
                    if best_metric.saturating_add(penalty) < current_metric {
                        overcome = true;
                    }
                    changed_plan.entry(*region_id).or_insert_with(BTreeMap::new).insert(gid, best_gw.id);
                }
            }
        }

        if overcome {
            Some(Downlink {
                action: MessageType::Update,
                version: self.version + 1,
                ack: self.version,
                via: changed_via,
                plan: changed_plan,
            })
        } else {
            None
        }
    }

    pub fn route_metric(&self, to: &Router, via: &Router, routers: &BTreeMap<RouterID, Router>) -> i32 {
        assert!(self != to);
        assert!(self != via);

        let mut result: Quality = Default::default();
        let mut route = vec![self];
        let mut current = self;

        while current != to {
            let next = if current == self { via } else { &routers[&current.via[&to.id]] };
            let quality = next.peers.get(&current.id);
            if quality.is_none() || quality.unwrap().reliability == 0 || !next.is_online() || route.contains(&next) {
                return i32::MAX;
            }
            result.concat(quality.unwrap(), DATABASE.connections[&current.id][&next.id].metric);

            route.push(next);
            current = next;
        }
        result.metric()
    }
}
