use crate::api::create_app;
use crate::router::Router;
use crate::settings::{CONFIG, TIMEOUT};
use crate::shared::data::{DATABASE, RouterID};
use crate::shared::protocol::{Downlink, Uplink};
use anyhow::{Context, Result};
use net::SocketAddr;
use std::collections::BTreeMap;
use std::net;
use std::sync::Arc;
use tokio::net::UdpSocket;
use tokio::sync::RwLock;
use tokio::time::Instant;

mod api;
mod quality;
mod router;
mod settings;
mod shared;

#[derive(Default, Debug)]
pub struct UpdatingState {
    router_id: RouterID,
    message: Downlink,
}

#[tokio::main]
async fn main() -> Result<()> {
    tracing_subscriber::fmt::init();

    let routers: BTreeMap<RouterID, Router> = DATABASE.routers.iter().map(|c| (c.id, Router::new(c.id))).collect();
    let routers = Arc::new(RwLock::new(routers));

    let listener = tokio::net::TcpListener::bind(CONFIG.http_bind).await?;
    let app = create_app(routers.clone());

    tokio::spawn(async move {
        println!("HTTP listening on {}", &listener.local_addr().unwrap());
        axum::serve(listener, app).await.unwrap();
    });

    let mut updating: UpdatingState = UpdatingState::default();

    let socket = UdpSocket::bind(CONFIG.udp_bind).await?;
    println!("UDP listening on {}", CONFIG.udp_bind);
    let mut buf = [0u8; u16::MAX as usize]; // Max UDP size
    let start = Instant::now();

    loop {
        let (len, addr) = socket.recv_from(&mut buf).await?;
        if let Ok((mut uplink, _)) = bincode::decode_from_slice::<Uplink, _>(&buf[..len], bincode::config::standard()) {
            let mut routers = routers.write().await;
            let now = Instant::now();
            // 将超时的路由器下线
            for router in routers.values_mut() {
                if router.is_online() && router.last_seen.duration_since(now) >= TIMEOUT {
                    router.offline();
                }
            }
            if updating.router_id != Default::default() && !routers.get(&updating.router_id).context("router not found")?.is_online() {
                updating.router_id = Default::default();
            }

            tracing::debug!("recv {:?}", uplink);
            // 处理收到的消息
            if let Some(router) = routers.get_mut(&uplink.id)
                && let Some(downlink) = router.on_message(&mut uplink, addr, &mut updating, now)
            {
                tracing::debug!("sync to {:?}: {:?}", router.id, downlink);
                send(&downlink, &mut buf, &socket, &addr).await?;
            } else if updating.router_id == Default::default()
                && now.duration_since(start) >= TIMEOUT // 刚启动时静默，先学习
                && let Some(router) = routers.get(&uplink.id)
                && router.is_online()
                && router.last_update != now
                && let Some(downlink) = router.update(now, &routers)
            {
                updating.router_id = router.id;
                updating.message = downlink;
                tracing::info!("command to {:?}: {:?}", router.id, updating.message);
                send(&updating.message, &mut buf, &socket, &addr).await?;
                routers.get_mut(&uplink.id).context("router not found")?.version += 1;
            }
        }
    }
}

async fn send(downlink: &Downlink, buf: &mut [u8], socket: &UdpSocket, addr: &SocketAddr) -> Result<()> {
    let len = bincode::encode_into_slice(downlink, buf, bincode::config::standard())?;
    let _ = socket.send_to(&buf[..len], addr).await;
    Ok(())
}
