import { RemoteInfo, Socket } from 'dgram';
import { DownloadMessage, PeerQuality, UploadMessage } from '../protocol';
import routers from '../import/data/Router.json';
// import plans from '../config/plans.json';
import assert from 'assert';
import { Quality } from './Quality';
import _ from 'lodash';
import config from '../config/config.json';

import _connections from '../import/connections.json';

const connections: Record<number, Record<number, { metric: number, protocol: string }>> = _connections;

export class Router {
  static updating?: Router;
  static updating_timer: any;
  static timeout_timer: any;

  static all: Router[] = routers.map(s => new Router(s.id));

  // config
  // id!: number;

  seq = 0;
  peers: Record<number, PeerQuality> = {};
  via: Map<Router, Router> = new Map();
  // plan: Record<number, number> = {};

  rinfo?: RemoteInfo;

  constructor(public id: number) {
    // Object.assign(this, config);
  }

  reset() {
    this.seq = 0;
    this.peers = {};
    for (const router of Router.all.filter(r => r.id !== this.id)) {
      this.via.set(router, router);
    }
    // for (const plan of plans.filter(plan => !plan.routers.includes(this.id))) {
    //   this.plan[plan.id] = this.id;
    // }
  }

  lost() {
    this.seq = 0;
    this.rinfo = undefined;
    return console.log(`router ${this.id} lost connection.`);
  }

  route_quality(to: Router, via: Router = this.via.get(to)!) {
    assert(via !== undefined);
    assert(this !== via);
    assert(this !== to);

    const result = new Quality();

    let current: Router = this;
    let next = via;
    const route = [current, next];

    while (true) {
      const quality = next.peers[current.id];
      if (!quality || quality.reliability <= 0) return Quality.unreachable; // 不通
      result.concat(quality.delay, quality.jitter, quality.reliability, connections[current.id][next.id].metric);
      if (next === to) return result; // 到达

      // 寻找下一跳
      current = next;
      next = current.via.get(to)!;
      assert(next); //to server_id 型路由，由于 server 两两相连，下一跳一定是存在的，至少能直达
      if (route.includes(next)) {
        // 环路
        return Quality.unreachable;
      } else {
        route.push(next);
      }
    }
  }

  onMessage(hello: UploadMessage) {
    if (hello.ack === 0) this.reset();
    if (hello.peers) this.peers = hello.peers;
    if (Router.updating === this && hello.ack === this.seq + 1) {
      clearInterval(Router.updating_timer);
      clearTimeout(Router.timeout_timer);
      Router.updating = undefined;
      this.seq++;
    }
  }

  update(socket: Socket) {
    const changedVia: Record<number, number> = {};
    const metric: Record<number, number> = {};
    for (const to of Router.all.filter(r => r.id !== this.id)) {
      // 计算最优下一跳

      const items: [Router, number][] = Router.all.filter(r => r.id !== this.id).map(r => [r, this.route_quality(to, r).metric()]);
      const [currentRoute, currentMetric] = items.find(([v, m]) => v === this.via.get(to))!;
      const [bestRoute, bestMetric] = _.minBy(items, (([v, m]) => m))!;

      // 变更
      if (currentRoute !== bestRoute && bestMetric + config.throttle < currentMetric) {
        changedVia[to.id] = bestRoute.id;
        this.via.set(to, bestRoute);
        metric[to.id] = bestMetric;
        // console.log(`change: this ${this.id} to ${to.id} current ${this.via.get(to)!.id} quality ${JSON.stringify(this.route_quality(to, this.via.get(to)!))} metric ${this.route_quality(to, this.via.get(to)!).metric()} best ${best_route.id} quality ${JSON.stringify(this.route_quality(to, best_route))} metric ${this.route_quality(to, best_route).metric()}`);
      } else {
        metric[to.id] = currentMetric;
      }
    }


    // 计算 route plan
    // 凡是自己可以作为那个 plan 出口的，是不会计算直接跳过的，所以这里有 plan 到自己的意思是，没有找到任何一个通的可以出的地方，所以只好从自己出了
    const changedPlan: Record<number, number> = {};
    // for (const plan of plans.filter(plan => !plan.routers.includes(this.id))) {
    //   const currentPlan = this.plan[plan.id];
    //   const currentMetric = currentPlan === this.id ? Infinity : metric[currentPlan];
    //   const items = plan.routers.map(to => [to, metric[to]]);
    //   const [bestPlan, bestMetric] = _.minBy(items, ([t, m]) => m)!;
    //
    //   if (currentPlan !== this.id && bestMetric === Infinity) {
    //     // 原来通的，现在不通了
    //     this.plan[plan.id] = changedPlan[plan.id] = this.id;
    //   } else if (currentPlan !== bestPlan && bestMetric + config.throttle < currentMetric) {
    //     this.plan[plan.id] = changedPlan[plan.id] = bestPlan;
    //   }
    // }

    if (!_.isEmpty(changedVia) || !_.isEmpty(changedPlan)) {
      Router.updating = this;
      Router.updating_timer = setInterval(() => this.send(socket, changedVia, changedPlan), config.interval);
      this.send(socket, changedVia, changedPlan);
      Router.timeout_timer = setTimeout(() => {
        this.lost();
        clearInterval(Router.updating_timer);
        Router.updating = undefined;
      }, config.timeout * config.interval);
    }
  }

  send(socket: Socket, via: Record<number, number>, plan: Record<number, number>) {
    if (!this.rinfo) return;
    const message: DownloadMessage = { seq: this.seq, via, plan };
    console.log(this.id, message);
    return socket.send(JSON.stringify(message), this.rinfo.port, this.rinfo.address);
  }
}

for (const router of Router.all) {
  router.reset();
}
