use crate::data::{ConnectionData, RouterData};
use crate::protocol::{Change, PeerQuality, Report};
use crate::quality::Quality;
use crate::settings::THROTTLE;
use crate::UpdatingState;
use serde::Serialize;
use std::collections::BTreeMap;
use std::net::SocketAddr;
use tokio::time::Instant;

#[derive(Serialize, Clone)]
pub struct Router {
    pub id: u8,
    #[serde(skip)]
    pub seq: u8,
    // quality from peer to self. HashMap 的 key 是 from.
    pub peers: BTreeMap<u8, PeerQuality>,
    pub via: BTreeMap<u8, u8>,                // dst router_id -> next hop router_id
    pub plan: BTreeMap<u8, BTreeMap<u8, u8>>, // group id -> region id -> gateway_id
    #[serde(skip)]
    pub time: Instant,      // ms
    #[serde(skip)]
    pub addr: Option<SocketAddr>,
    // Static config
    // pub config: RouterData,
}

impl PartialEq<Self> for Router {
    fn eq(&self, other: &Self) -> bool {
        self.id == other.id
    }
}

impl Eq for Router {}

impl Router {
    pub fn new(data: &RouterData, routers: &Vec<RouterData>, connections: &BTreeMap<u8, BTreeMap<u8, ConnectionData>>) -> Self {
        Self {
            id: data.id,
            seq: 0,
            peers: connections
                .iter()
                .filter(|(_, to)| to.contains_key(&data.id))
                .map(|(&from, _)| {
                    (
                        from,
                        PeerQuality {
                            delay: 0,
                            reliability: 0,
                            jitter: 0,
                        },
                    )
                })
                .collect(),
            via: routers.iter().filter(|r| r.id != data.id).map(|r| (r.id, r.id)).collect(),
            plan: BTreeMap::new(),
            time: Instant::now(),
            addr: None,
        }
        // router.reset(all_router_ids);
        // router
    }

    pub fn offline(&mut self) {
        if self.addr != None {
            println!("router {} offline", self.id);
        }
        self.addr = None;
        for peer in self.peers.values_mut() {
            peer.reliability = 0;
        }
    }
    pub fn online(&mut self, addr: SocketAddr) {
        if self.addr == None {
            println!("router {} online", self.id);
        }
        self.addr = Some(addr);
        self.time = Instant::now();
    }

    pub fn on_message(&mut self, data: Report, addr: SocketAddr, updating: &mut UpdatingState) -> Option<Change> {
        // 不带 syn, 序号正确：已经在线的，或者是上线流程走完
        // 有 syn：客户端新上线；无 addr, 服务端新上线新上线，先把客户端标记为离线来走登录流程
        if data.syn || self.addr == None {
            self.offline();
            self.finish(updating);
            Some(Change {
                seq: self.seq,
                via: self.via.clone(),
                plan: self.plan.clone(),
                rst: true,
            })
        } else if data.ack == self.seq.wrapping_add(1) {
            self.online(addr);
            for (current, new) in self.peers.values_mut().zip(data.peers) {
                *current = new
            }
            if (updating.router_id == self.id) {
                self.via.append(&mut updating.change.via);
                self.finish(updating);
            }
            None
        } else {
            // 序号不对的，忽略
            None
        }
    }

    pub fn finish(&self, updating: &mut UpdatingState) {
        if updating.router_id == self.id {
            updating.router_id = 0;
        }
    }

    pub fn update(&self, routers: &BTreeMap<u8, Router>, connections: &BTreeMap<u8, BTreeMap<u8, ConnectionData>>) -> Option<Change> {
        if self.addr.is_none() {
            return None;
        }

        let mut changed_via: BTreeMap<u8, u8> = BTreeMap::new();
        // let mut metric: BTreeMap<u8, i32> = BTreeMap::new();

        for to in routers.values().filter(|&r| r != self) {
            let candidate: Vec<(&Router, i32)> = connections[&self.id]
                .keys()
                .map(|id| routers.get(id).unwrap())
                .map(|r| (r, self.route_quality(to, r, routers, connections).metric()))
                .collect();
            let current = candidate.iter().find(|(v, _)| v.id == *self.via.get(&to.id).expect(""));
            let (best_router, best_metric) = candidate.iter().min_by_key(|(_, m)| m).unwrap();
            if current.is_none_or(|(current_router, current_metric)| current_router != best_router && *best_metric != i32::MAX && *best_metric + THROTTLE < *current_metric) {
                changed_via.insert(to.id, best_router.id);
                // metric.insert(to.id, *best_metric);
            }
        }

        if changed_via.len() > 0 {
            Some(Change {
                seq: self.seq + 1,
                rst: false,
                via: changed_via,
                plan: BTreeMap::new(),
            })
        } else {
            None
        }
    }

    pub fn route_quality(&self, to: &Router, via: &Router, routers: &BTreeMap<u8, Router>, connections: &BTreeMap<u8, BTreeMap<u8, ConnectionData>>) -> Quality {
        assert!(self != to);
        assert!(self != via);

        let mut result: Quality = Quality::default();
        let mut route = vec![self, via];

        let mut current = self;
        let mut next = via;
        loop {
            // 不通的情况 via 会标记为直连，实际不可达
            match next.peers.get(&current.id) {
                None => return Quality::UNREACHABLE,
                Some(quality) => {
                    if quality.reliability == 0 {
                        // 不通
                        return Quality::UNREACHABLE;
                    }
                    result.concat(quality, connections[&current.id][&next.id].metric);

                    if next == to {
                        // 到达
                        return result;
                    }

                    // Next hop
                    current = next;
                    next = &routers[&current.via[&to.id]];

                    // 检查环路
                    if route.contains(&next) {
                        return Quality::UNREACHABLE;
                    }
                    route.push(next);
                }
            }
        }
    }
}
