use crate::protocol::PeerQuality;
use serde::{Deserialize, Serialize};

#[derive(Debug, Clone, Copy, Serialize, Deserialize)]
pub struct Quality {
    pub delay: i32,
    pub jitter: u32,
    pub reliability: f32,
    pub cost: u32,
}

impl Quality {
    pub const UNREACHABLE: Quality = Quality {
        delay: 0,
        jitter: 0,
        reliability: 0.0,
        cost: 0,
    };

    pub fn concat(&mut self, next: &PeerQuality, cost: u32) -> &mut Self {
        self.delay += next.delay as i32;
        self.jitter += next.jitter as u32;
        self.reliability *= next.reliability as f32 / 255.0;
        self.cost += cost;
        self
    }

    pub fn metric(&self) -> i32 {
        assert!(0.0 <= self.reliability && self.reliability <= 1.0);

        if self.reliability == 0.0 {
            i32::MAX
        } else {
            self.delay + ((1.0 - self.reliability) * 10000.0).round() as i32 + self.cost as i32
        }
    }

    pub(crate) fn default() -> Self {
        Self {
            delay: 0,
            jitter: 0,
            reliability: 1.0,
            cost: 0,
        }
    }
}
