use crate::api::create_app;
use crate::data::{ConnectionData, RouterData};
use crate::protocol::{Change, Report};
use crate::router::Router;
use crate::settings::{Settings, INTERVAL, TIMEOUT};
use anyhow::Result;
use ::config::Config;
use config::Environment;
use net::SocketAddr;
use serde::Serialize;
use std::collections::BTreeMap;
use std::net;
use std::ops::Bound::{Excluded, Included, Unbounded};
use std::sync::Arc;
use tokio::net::UdpSocket;
use tokio::select;
use tokio::sync::RwLock;
use tokio::time::interval;

mod api;
mod data;
mod gateway_group;
mod protocol;
mod quality;
mod router;
mod settings;

#[derive(Default, Serialize)]
pub struct UpdatingState {
    router_id: u8,
    change: Change,
}

#[tokio::main]
async fn main() -> Result<()> {
    let config: Settings = Config::builder().add_source(Environment::default()).build()?.try_deserialize()?;

    let routers_data: Arc<Vec<RouterData>> = Arc::new(serde_json::from_str(&std::fs::read_to_string("import/data/Router.json")?)?);
    let connections = Arc::new(serde_json::from_str(&std::fs::read_to_string("import/connections.json")?)?);

    // let routers_map: BTreeMap<u8, RouterData> = routers_data.iter().map(|r| (r.id, r.clone())).collect();
    // let routers_map = Arc::new(routers_map);

    let routers: BTreeMap<u8, Router> = routers_data.iter().map(|c| (c.id, Router::new(c, &routers_data, &connections))).collect();
    let routers = Arc::new(RwLock::new(routers));
    // let gateway_groups_data: Vec<GatewayGroup> = serde_json::from_str(&std::fs::read_to_string(
    //     "../import/data/GatewayGroup.json",
    // )?)?;
    // 3. Initialize State
    // let all_router_ids: Vec<u8> = routers_data.iter().map(|r| r.id).collect();
    // let mut gateway_groups_map: BTreeMap<u8, GatewayGroup> = BTreeMap::new();
    // let raw_groups = gateway_groups_data.clone();
    // for g_data in gateway_groups_data {
    //     let g = GatewayGroup::new(g_data, &raw_groups, &routers_data);
    //     gateway_groups_map.insert(g.id, g);
    // }

    let mut updating: UpdatingState = UpdatingState::default();

    let listener = tokio::net::TcpListener::bind(config.http_bind).await?;
    let app = create_app(routers_data.clone(), connections.clone(), routers.clone());

    tokio::spawn(async move {
        println!("HTTP listening on {}", &listener.local_addr().unwrap());
        axum::serve(listener, app).await.unwrap();
    });

    let socket = UdpSocket::bind(config.udp_bind).await?;
    println!("UDP listening on {}", config.udp_bind);
    let mut buf = [0u8; u16::MAX as usize]; // Max UDP size

    let mut interval = interval(INTERVAL);
    let mut cursor = 0;

    loop {
        select! {
            biased;

            result = socket.recv_from(&mut buf) => {
                let (len, addr) = result?;
                if let Ok((report, _)) = bincode::decode_from_slice::<Report, _>(&buf[..len], bincode::config::standard()) {
                    let mut routers = routers.write().await;
                    if let Some(router) = routers.get_mut(&report.id) {
                        let old_updating_exists = updating.router_id != 0;
                        if let Some(change) = router.on_message(report, addr, &mut updating) {
                            // 新上线的客户端，发送 rst，seq 不增加
                            send(&change, &mut buf, &socket, &router.addr.unwrap()).await?;
                        }
                        if old_updating_exists && updating.router_id == 0 { // an updating just resolved
                            tick(&mut routers, &mut cursor, &socket, &mut buf, &mut updating, &connections).await?
                        }
                    }
                }
            }

            now = interval.tick() => {
                let mut routers = routers.write().await;
                if updating.router_id != 0 {
                    let router = routers.get_mut(&updating.router_id).expect("updating router_id should exist");
                    if now.duration_since(router.time) < TIMEOUT {
                        send(&updating.change, &mut buf, &socket, &router.addr.unwrap()).await?;
                    } else {
                        router.offline();
                        router.finish(&mut updating);
                    }
                }
                if updating.router_id == 0 {
                    for router in routers.values_mut() {
                        if router.addr != None && now.duration_since(router.time) > TIMEOUT {
                            router.offline();
                        }
                    }
                    tick(&mut routers, &mut cursor, &socket, &mut buf, &mut updating, &connections).await?
                }
            }
        }
    }

    async fn tick(
        routers: &mut BTreeMap<u8, Router>,
        cursor: &mut u8,
        socket: &UdpSocket,
        buf: &mut [u8],
        updating: &mut UpdatingState,
        connections: &BTreeMap<u8, BTreeMap<u8, ConnectionData>>,
    ) -> Result<()> {
        for (_, router) in routers.range((Excluded(*cursor), Unbounded)).chain(routers.range((Unbounded, Included(*cursor)))) {
            if let Some(change) = router.update(&routers, &connections) {
                *cursor = router.id;
                updating.router_id = router.id;
                updating.change = change;
                send(&updating.change, buf, socket, &router.addr.unwrap()).await?;
                break;
            }
        }
        if updating.router_id != 0 {
            let router = routers.get_mut(&updating.router_id).unwrap();
            router.seq += 1;
        }
        Ok(())
    }

    async fn send(change: &Change, buf: &mut [u8], socket: &UdpSocket, addr: &SocketAddr) -> Result<()> {
        let len = bincode::encode_into_slice(change, buf, bincode::config::standard())?;
        let _ = socket.send_to(&buf[..len], addr).await;
        Ok(())
    }
}
