import { Component, OnInit } from '@angular/core';
import { map } from 'rxjs/operators';
import { ApiService, RouterStatus } from '../api.service';
import * as _ from 'lodash-es';
import { Quality } from '../Quality';

@Component({
  selector: 'app-dashboard',
  templateUrl: './dashboard.component.html',
  styleUrls: ['./dashboard.component.scss']
  // changeDetection: ChangeDetectionStrategy.OnPush
})
export class DashboardComponent implements OnInit {
  // routers: RouterStatus[] = [];
  displayedColumns: string[] = [];
  links = this.api.state().pipe(
    map(routers => routers.map(router =>
      ({
        id: router.id,
        name: this.api.routers.find(r => r.id == router.id)!.name,
        peers: _.mapValues(router.peers, (quality, id1) => this.displayQuality(routers, router, parseInt(id1)))
      })
    ))
  );
  routed = false;

  constructor(public api: ApiService) {
  }

  displayQuality(routers: RouterStatus[], from: RouterStatus, toId: number) {
    const to = routers.find(r => r.id === toId);
    if (!to) return;

    const quality = new Quality();

    if (this.routed) {
      quality
        .concatQuality(this.calculate(routers, from, to))
        .concatQuality(this.calculate(routers, to, from));

      return {
        delay: quality.reliability <= 0.01 ? 'x' : quality.delay,
        color: `hsl(${this.mapNumber(quality.reliability, 0.9, 1, 0, 120)}, 100%, 40%)`,
        decoration: quality.hop <= 2 ? '' : `underline hsl(${this.mapNumber(quality.hop, 4, 8, 120, 0)}, 100%, 40%)`
      };
    } else {
      const link = this.api.connections[from.id]?.[toId];
      if (!link) return;
      if (!to.peers[from.id]) return;

      quality
        .concatPeer(from.peers[toId])
        .concatPeer(to.peers[from.id]);

      const decorations: Record<string, string> = {
        oc: 'underline blue',
        wg: 'underline red',
        tun: 'underline green'
      };

      return {
        delay: quality.reliability <= 0.01 ? 'x' : quality.delay,
        color: `hsl(${this.mapNumber(quality.reliability, 0.9, 1, 0, 120)}, 100%, 40%)`,
        // decoration: decorations[link.protocol],
        protocol: link.protocol.slice(0, 1)
      };
    }
  }

  mapNumber(value: number, minFrom: number, maxFrom: number, minTo: number, maxTo: number) {
    return minTo + (maxTo - minTo) * ((_.clamp(value, minFrom, maxFrom) - minFrom) / (maxFrom - minFrom));
  }

  async ngOnInit() {
    this.displayedColumns = ['name', 'id'].concat(this.api.routers.map(r => r.name));
  }

  trackById(index: number, item: any) {
    return index;
  }

  /** Based on the screen size, switch from standard to one column per row */

  private calculate(routers: RouterStatus[], from: RouterStatus, to: RouterStatus): Quality {
    const result = new Quality();

    let current = from;
    while (current !== to) {
      const next = routers.find(r => r.id == current.via[to.id])!;
      result.concatPeer(next.peers[current.id]);
      current = next;
    }

    return result;
  }
}
