import { RemoteInfo, Socket } from 'dgram';
import { Change, PeerQuality, Report } from '../protocol';
import routers from '../import/data/Router.json';
// import plans from '../config/plans.json';
import assert from 'assert';
import { Quality } from './Quality';
import _ from 'lodash';
import config from '../config/config.json';

import _connections from '../import/connections.json';
import { GatewayGroup } from '../import/scripts/GatewayGroup';

const connections: Record<number, Record<number, { metric: number; protocol: string }>> = _connections;

export class Router {

  static all: Router[] = routers.map((s) => new Router(s.id));
  static updating?: {
    router: Router;
    message: Change;
  };
  static groups: Record<number, Router[]>;

  seq = 0;
  peers: Record<number, PeerQuality> = {};
  via: Map<Router, Router> = new Map();
  plan: Record<number, number> = {};

  time: number = 0;
  rinfo?: RemoteInfo;

  constructor(public id: number) {}

  static update(socket: Socket) {
    for (const from of Router.all) from.update(socket);
  }

  reset() {
    this.seq = 0;
    this.peers = {};
    for (const router of Router.all.filter((r) => r.id !== this.id)) {
      this.via.set(router, router);
    }
    if (Router.updating?.router == this) Router.updating = undefined;
    // for (const plan of plans.filter(plan => !plan.routers.includes(this.id))) {
    //   this.plan[plan.id] = this.id;
    // }
  }

  onMessage(socket: Socket, data: Report) {
    // console.log(`recv: ${this.id} ${JSON.stringify(data)} this.seq:${this.seq}`);
    if (data.ack == this.seq + 1) {
      this.time = Date.now();
      if (data.peers) this.peers = data.peers;
      if (Router.updating?.router === this) {
        Router.updating = undefined;
        Router.update(socket);
      }
    } else if (data.ack === 0) {
      // 客户端重启
      this.reset();
      this.time = Date.now();
      this.send(socket, { seq: this.seq, via: {}, plan: {} });
    } else if (this.seq == 0) {
      // 服务器重启或客户端下线
      this.time = Date.now();
      this.send(socket, { seq: this.seq, via: {}, plan: {} });
    } else {
      console.log(`ignoring packet from ${data.id}, packet ack=${data.ack}, server seq=${this.seq}`);
    }
  }

  update(socket: Socket) {
    if (!this.rinfo) return;
    const timeout = Router.updating?.router == this ? config.timeout : config.timeout2;
    if (Date.now() - this.time > timeout * config.interval) {
      console.log(`router ${this.id} lost connection.`);
      this.rinfo = undefined;
      this.reset();
      return;
    }
    if (Router.updating) {
      if (Router.updating.router == this) this.send(socket, Router.updating.message);
      return;
    }

    const changedVia: Record<number, number> = {};
    const metric: Record<number, number> = {};
    for (const to of Router.all.filter((r) => r.id !== this.id)) {
      // 计算最优下一跳

      const items: [Router, number][] = Router.all.filter((r) => r.id !== this.id).map((r) => [r, this.route_quality(to, r).metric()]);
      const [currentRoute, currentMetric] = items.find(([v, m]) => v === this.via.get(to))!;
      const [bestRoute, bestMetric] = _.minBy(items, ([v, m]) => m)!;

      // 变更
      if (currentRoute !== bestRoute && bestMetric + config.throttle < currentMetric) {
        changedVia[to.id] = bestRoute.id;
        this.via.set(to, bestRoute);
        metric[to.id] = bestMetric;
        // console.log(`change: this ${this.id} to ${to.id} current ${this.via.get(to)!.id} quality ${JSON.stringify(this.route_quality(to, this.via.get(to)!))} metric ${this.route_quality(to, this.via.get(to)!).metric()} best ${best_route.id} quality ${JSON.stringify(this.route_quality(to, best_route))} metric ${this.route_quality(to, best_route).metric()}`);
      } else {
        metric[to.id] = currentMetric;
      }
    }

    // 计算 route plan
    // 凡是自己可以作为那个 plan 出口的，是不会计算直接跳过的，所以这里有 plan 到自己的意思是，没有找到任何一个通的可以出的地方，所以只好从自己出了
    const changedPlan: Record<number, number> = {};
    for (const [_groupId, groupRouters] of Object.entries(Router.groups).filter(([_, g]) => !g.includes(this))) {
      const groupId = parseInt(_groupId);
      const currentPlan = this.plan[groupId];
      const currentMetric = currentPlan === this.id ? Infinity : metric[currentPlan];
      const items = groupRouters.map((to) => [to.id, metric[to.id]]);
      const [bestPlan, bestMetric] = _.minBy(items, ([_, m]) => m)!;

      if (currentPlan !== this.id && bestMetric === Infinity) {
        // 原来通的，现在不通了
        this.plan[groupId] = changedPlan[groupId] = this.id;
      } else if (currentPlan !== bestPlan && bestMetric + config.throttle < currentMetric) {
        this.plan[groupId] = changedPlan[groupId] = bestPlan;
      }
    }

    if (!_.isEmpty(changedVia) || !_.isEmpty(changedPlan)) {
      this.seq++;
      const message: Change = { seq: this.seq, via: changedVia, plan: changedPlan };
      Router.updating = { router: this, message };
      this.send(socket, message);
    }
  }

  send(socket: Socket, message: Change) {
    console.log(`send: ${this.id} ${JSON.stringify(message)}`);
    return socket.send(JSON.stringify(message), this.rinfo!.port, this.rinfo!.address);
  }

  route_quality(to: Router, via: Router = this.via.get(to)!) {
    assert(via !== undefined);
    assert(this !== via);
    assert(this !== to);

    const result = new Quality();

    let current: Router = this;
    let next = via;
    const route = [current, next];

    while (true) {
      const quality = next.peers[current.id];
      if (!quality || quality.reliability <= 0) return Quality.unreachable; // 不通
      result.concat(quality.delay, quality.jitter, quality.reliability, connections[current.id][next.id].metric);
      if (next === to) return result; // 到达

      // 寻找下一跳
      current = next;
      next = current.via.get(to)!;
      assert(next); //to server_id 型路由，由于 server 两两相连，下一跳一定是存在的，至少能直达
      if (route.includes(next)) {
        // 环路
        return Quality.unreachable;
      } else {
        route.push(next);
      }
    }
  }
}

for (const router of Router.all) router.reset();

function groupRouters(g: GatewayGroup): Router[] {
  return _.uniq(
    g.locationPrefix
      .flatMap((p) => routers.filter((r) => r.location.startsWith(p)))
      .concat(routers.filter((r) => g.includeRouters.includes(r.name)))
      .filter((r) => !g.excludeRouters.includes(r.name))
      .map((r) => Router.all.find((r1) => r1.id === r.id)!)
      .concat(gatewayGroups.filter((g1) => g.children.includes(g1.name)).flatMap((c) => groupRouters(c)))
  );
}

Router.groups = Object.fromEntries(gatewayGroups.map((g) => [g.id, groupRouters(g)]));
console.log(Router.groups);
