import dgram from 'dgram';
import http from 'http';

import { DownloadMessage, UploadMessage } from '../protocol';
import assert from 'assert';
import { Router } from './Router';
import _ from 'lodash';

import config from '../config/config.json';

let updating: any = null;
let timeout_timer: any = null;

export function send_route(from: Router, to: Router, via: Router) {
  const message: DownloadMessage = { seq: from.seq, to: to.id, via: via.id };
  console.log(from.id, message);
  return socket.send(JSON.stringify(message), from.rinfo!.port, from.rinfo!.address);
}

const socket = dgram
  .createSocket('udp4')
  .on('listening', () => {
    const address = socket.address();
    console.log(`listening ${address.address}:${address.port}`);
  })
  .on('message', function(message, rinfo) {
    try {
      const hello: UploadMessage = JSON.parse(message.toString());

      assert(hello.id);
      const router: Router = Router.all.find(r => r.id === hello.id)!;
      assert(router);

      router.rinfo = rinfo;

      if (hello.ack === 0) {
        router.reset();
      }

      if (hello.peers) router.peers = hello.peers;

      if (updating) {
        if (hello.ack === router.seq + 1) {
          clearInterval(updating);
          clearTimeout(timeout_timer);
          updating = null;
          router.seq++;
        }
        return;
      }
    } catch (e) {
      console.warn(e);
    }

    for (const from of Router.all) {
      for (const to of Router.all.filter(r => r.id !== from.id)) {
        // 计算最优下一跳
        const best_route = _.minBy(Router.all.filter(router => router.id !== from.id), (router) => from.route_quality(to, router).metric())!;

        // 变更
        if (from.via.get(to) !== best_route && from.route_quality(to, best_route).metric() + 1 < from.route_quality(to, from.via.get(to)!).metric()) {
          console.log(`change: from ${from.id} to ${to.id} current ${from.via.get(to)!.id} quality ${JSON.stringify(from.route_quality(to, from.via.get(to)!))} metric ${from.route_quality(to, from.via.get(to)!).metric()} best ${best_route.id} quality ${JSON.stringify(from.route_quality(to, best_route))} metric ${from.route_quality(to, best_route).metric()}`);
          from.via.set(to, best_route);
          updating = setInterval(() => send_route(from, to, best_route), config.interval);
          send_route(from, to, best_route);
          timeout_timer = setTimeout(function() {
            from.lost();
            clearInterval(updating);
            updating = null;
          }, config.timeout * config.interval);
          return;
        }
      }
    }
  });
socket.bind(config.port);

//web management api
http
  .createServer(function(req, res) {
    res.writeHead(200, { 'Content-Type': 'application/json' });
    const result: Record<number, Record<number, number>> = Object.fromEntries(Router.all.map(from => [from.id, Object.fromEntries([...from.via.entries()].map(([to, via]) => [to.id, via.id]))]));
    return res.end(JSON.stringify(result));
  })
  .on('listening', () => {
    const address = socket.address();
    console.log(`http listening ${address.address}:${address.port}`);
  })
  .listen(config.port);
