import { RemoteInfo } from 'dgram';
import { PeerQuality } from '../protocol';
import routers from '../config/routers.json';
import assert from 'assert';
import { Quality } from './Quality';

export class Router {
  static all: Router[] = routers.map(s => Object.assign(new Router(), s));

  // config
  id!: number;
  inbound_cost!: number;
  outbound_cost!: number;

  seq = 0;
  peers: Record<number, PeerQuality> = {};
  via: Map<Router, Router> = new Map();

  rinfo?: RemoteInfo;

  reset() {
    this.seq = 0;
    this.peers = {};
    for (const router of Router.all.filter(r => r.id !== this.id)) {
      this.via.set(router, router);
    }
  }

  lost() {
    this.seq = 0;
    this.rinfo = undefined;
    return console.log(`router ${this.id} lost connection.`);
  }

  route_quality(to: Router, via: Router) {
    assert(via != null);
    assert(this !== via);
    assert(this !== to);

    const result = new Quality();

    let current: Router = this;
    let next = via;
    const route = [current, next];

    while (true) {
      const quality = next.peers[current.id];
      if (!quality || quality.reliability <= 0) return Quality.unreachable; // 不通
      result.concat(quality.delay, quality.jitter, quality.reliability, current.outbound_cost + next.inbound_cost);
      if (next === to) return result; // 到达

      // 寻找下一跳
      current = next;
      next = current.via.get(to)!;
      assert(next); //to server_id 型路由，由于 server 两两相连，下一跳一定是存在的，至少能直达
      if (route.includes(next)) {
        // 环路
        return Quality.unreachable;
      } else {
        route.push(next);
      }
    }
  }
}

for (const router of Router.all) {
  router.reset();
}
