use hickory_resolver::Resolver;
use hickory_resolver::name_server::GenericConnector;
use hickory_resolver::proto::runtime::TokioRuntimeProvider;
use netlink_packet_route::route::RouteProtocol;
use serde::Deserialize;
use std::net::SocketAddr;
use std::time::Duration;

#[derive(Deserialize)]
pub struct Settings {
    pub id: u8,
    pub server: Endpoint,
    pub bind: SocketAddr,
    // pub TIMEOUT: u32,
    // pub history: usize,
    // pub INTERVAL: u64,
    // pub table: u16,
    // pub proto: u16,
}

#[derive(Deserialize)]
#[serde(try_from = "String")]
pub struct Endpoint {
    pub host: String,
    pub port: u16,
}

impl TryFrom<String> for Endpoint {
    type Error = String;

    fn try_from(value: String) -> Result<Self, Self::Error> {
        let parts: Vec<&str> = value.rsplitn(2, ':').collect();
        if parts.len() != 2 {
            return Err(format!("Invalid endpoint format: {}", value));
        }

        let port = parts[0].parse::<u16>().map_err(|e| format!("Invalid port: {}", e))?;
        let host = parts[1].to_string();

        Ok(Endpoint { host, port })
    }
}

impl Endpoint {
    pub async fn to_socket_addrs(&self, resolver: &Resolver<GenericConnector<TokioRuntimeProvider>>) -> anyhow::Result<SocketAddr> {
        let lookup = resolver.lookup_ip(&self.host).await?;
        let ip = lookup.into_iter().next().ok_or_else(|| anyhow::anyhow!("No IP address found for host"))?;
        Ok(SocketAddr::new(ip, self.port))
    }
}

pub const INTERVAL: Duration = Duration::from_secs(1);
pub const TIMEOUT: Duration = Duration::from_secs(60);
pub const HISTORY: u8 = 100;
pub const ROUTE_PROTOCOL: RouteProtocol = RouteProtocol::Other(252);
