use crate::connection::Connection;
use crate::data::Router as RouterData;
use crate::protocol::{Change, Hello, Report};
use crate::router::Router;
use crate::server::Server;
use crate::settings::{INTERVAL, Settings};
use config::Config;
use std::collections::BTreeMap;
use std::fs;
use std::time::SystemTime;
use tokio::net::UdpSocket;
use tokio::time;

mod connection;
mod data;
mod gateway_group;
mod protocol;
mod router;
mod server;
mod settings;
use hickory_resolver::Resolver;

#[tokio::main]
async fn main() -> anyhow::Result<()> {
    let config: Settings = Config::builder().add_source(config::Environment::default()).build()?.try_deserialize()?;
    let routers_data = serde_json::from_slice::<Vec<RouterData>>(&fs::read("import/data/Router.json")?)?;
    let mut routers: BTreeMap<u8, Router> = routers_data.iter().map(|r| (r.id, Router::new(r, &config))).collect();
    let connections = serde_json::from_slice::<BTreeMap<u8, BTreeMap<u8, Connection>>>(&fs::read("import/connections.json")?)?;
    // let groups: Vec<GatewayGroup> = serde_json::from_slice(&fs::read("import/GatewayGroup.json")?)?;

    let mut server = Server::new(
        // &routers,
        // groups
        //     .iter()
        //     .map(|g| (g.id, g.routers(&groups, &routers_data)))
        //     .collect::<BTreeMap<u32, HashSet<u8>>>(),
    );

    let mut hello = Hello { time: 0 };

    let socket = UdpSocket::bind(config.bind).await?;

    let mut timer = time::interval(INTERVAL);
    let mut buf = [0; 1500];

    let mut syn: bool = true;

    let resolver = Resolver::builder_tokio()?.build();

    loop {
        let server_addr = config.server.to_socket_addrs(&resolver).await?;

        tokio::select! {
            biased; // 优先处理上面的

            result = socket.recv_from(&mut buf) => {
                let (len, addr) = result?;
                if addr == server_addr {
                    // from server
                    let (message, _): (Change, usize) = bincode::decode_from_slice(&buf[..len], bincode::config::standard())?;
                    server.on_message(&message, &routers_data, &connections[&config.id]).await;
                    syn = false;
                    let report = Report {
                        id: config.id,
                        syn,
                        ack: server.ack,
                        peers: Vec::new()
                    };
                    let message = bincode::encode_to_vec(&report, bincode::config::standard())?;
                    let _ = socket.send_to(message.as_slice(), server_addr).await;
                } else if let Some(peer) = Router::get(&mut routers, addr){
                    // from client
                    let (message, _): (Hello, usize) = bincode::decode_from_slice(&buf[..len], bincode::config::standard())?;
                    peer.on_message(&message);
                }
            }

            _ = timer.tick() => {
                // to clients
                hello.time = SystemTime::now().duration_since(SystemTime::UNIX_EPOCH)?.as_millis() as u32;
                let len = bincode::encode_into_slice(&hello, &mut buf, bincode::config::standard())?;
                for id in connections[&config.id].keys() {
                    let router = &routers[id];
                    let _ = socket.send_to(&buf[..len], router.link_address).await;
                }

                // to server
                let report = Report {
                    id: config.id,
                    syn,
                    ack: server.ack,
                    peers: connections
                        .iter()
                        .filter(|(_, to)| to.contains_key(&config.id))
                        .map(|(from,_)|routers.get_mut(from).unwrap().update(hello.time))
                        .collect(),
                };
                let len = bincode::encode_into_slice(&report, &mut buf, bincode::config::standard())?;
                let _ = socket.send_to(&buf[..len], server_addr).await;
            }
        }
    }
}
