import * as _ from 'lodash';
import config from '../config/config.json';
import { Hello, PeerQuality } from '../protocol';
import _routers from '../import/data/Router.json';
import subnets from '../import/data/Subnet.json';

import { Router as RouterData } from '../import/scripts/Router';

export interface Router extends RouterData {
}

const routers = <RouterData[]>_routers;

export class Router implements Hello, PeerQuality {
  static all: Record<number, Router> = _.keyBy(
    routers.filter((r) => r.id !== parseInt(process.env.RAILGUN_ID)).map((r) => new Router(r)),
    'id'
  );

  address: string;
  linkAddress: string;
  subnets: string[];

  delay: number = 0;
  jitter: number = 0;
  reliability: number = 0;
  seq: number = 0;
  time: number = 0;

  history: (number | undefined)[] = [];

  constructor(public data: RouterData) {
    Object.assign(this, data);
    this.linkAddress = `10.200.${data.id}.${parseInt(process.env.RAILGUN_ID)}`;
    this.subnets = subnets.filter((s) => s.router === data.name).map((s) => s.subnet);
  }

  reset() {
    this.delay = 0;
    this.reliability = 0;
    this.jitter = 0;
    this.seq = 0;
    this.time = 0;
    this.history = [];
  }

  onMessage(data: Hello) {
    // console.log(data);
    if (data.seq === 0 || data.seq < this.seq - config.timeout || data.seq > this.seq + config.timeout) {
      // 收到 seq = 0 或 seq 与之前差距较大，就 reset
      this.reset();
      this.seq = data.seq - 1;
    } else if (data.seq <= this.seq) {
      // 收到 seq 比已知略小的，忽略
      return;
    }
    // 全新或者 seq 比已知略大。

    const time = Date.now();
    const step = data.seq - this.seq;
    const delay = time - data.time;

    for (let i = 0; i < step - 1; i++) {
      this.history.push(undefined);
    }
    this.history.push(delay);
    this.history.splice(0, this.history.length - config.history);

    const history = this.history.filter((s) => s !== undefined);

    this.reliability = history.length / config.history;
    this.delay = _.mean(history) || 0;
    let jitterSum = 0;
    for (let i = 0; i < history.length - 1; i++) {
      jitterSum += Math.abs(history[i] - history[i + 1]);
    }
    this.jitter = jitterSum / (history.length - 1) || 0;

    this.seq = data.seq;
    this.time = time;
  }

  update(time: number): PeerQuality {
    if (this.reliability > 0) {
      // 有几个包没到
      const step = Math.floor((time - this.time) / config.interval);
      if (step > config.timeout) {
        this.reset();
      }
    }

    const lost = Math.max(0, (time - this.time) / config.interval - 2);

    return {
      delay: Math.round(this.delay),
      jitter: Math.round(this.jitter),
      reliability: Math.max(0, this.reliability - lost / config.history),
    };
  }
}
