import config from '../config/config.json';

import { parseInt } from 'lodash';
import { Router } from './Router';
import child_process from 'child_process';
import { GatewayGroup } from './GatewayGroup';

export class RouteWriter {
  static id = parseInt(process.env.RAILGUN_ID);
  static input: string[] = [];
  static via = new Map<number, number>();
  static plan = new Map<number, number>();

  static reset() {
    console.log('reset');
    this.input.push(`route flush table ${config.table} proto ${config.proto}`);
    for (const peer of Object.values(Router.all)) {
      this.setVia(peer.id, peer.id);
    }
    for (const group of Object.values(GatewayGroup.all).filter((g) => !g.routers.includes(this.id))) {
      this.setPlan(group, this.id);
    }
  }

  static setVia(toId: number, viaId: number) {
    this.via.set(toId, viaId);
    const to = Router.all[toId];
    const via = Router.all[viaId];
    // main
    for (const address of [to.address, ...to.subnets]) {
      this.input.push(`route replace ${address} via ${via.linkAddress} table ${config.table} proto ${config.proto}`);
    }
    // dest
    this.input.push(`route replace default via ${via.linkAddress} table ${to.destMark + 1000} proto ${config.proto}`);
    // plan
    for (const [table, to] of this.plan.entries()) {
      if (to === toId) {
        this.input.push(`route replace default via ${via.linkAddress} table ${table} proto ${config.proto}`);
      }
    }
  }

  static setPlan(group: GatewayGroup, toId: number) {
    const table = group.destMark + 1000;
    this.plan.set(table, toId);
    if (toId === this.id) {
      this.input.push(`route flush table ${table} proto ${config.proto}`);
    } else {
      const viaId = this.via.get(toId);
      const via = Router.all[viaId];
      this.input.push(`route replace default via ${via.linkAddress} table ${table} proto ${config.proto}`);
    }
  }

  static commit() {
    if (!this.input.length) return;
    // 可能改成异步的会更好点?
    const input = this.input.join('\n');
    console.log(input);
    try {
      child_process.execFileSync('ip', ['-force', '-batch', '-'], { input });
    } catch {
    }
    this.input = [];
  }
}
