use crate::data::Router as RouterData;
use crate::protocol::{Change, Hello, Report};
use crate::router::Router;
use crate::server::Server;
use crate::settings::{Settings, INTERVAL};
use anyhow::ensure;
use config::Config;
use std::collections::HashMap;
use std::fs;
use std::net::{IpAddr, Ipv6Addr, SocketAddr};
use std::time::SystemTime;
use tokio::net::UdpSocket;
use tokio::time;

mod data;
mod gateway_group;
mod protocol;
mod route_writer;
mod router;
mod server;
mod settings;

#[tokio::main]
async fn main() -> anyhow::Result<()> {
    let config: Settings = Config::builder()
        .add_source(config::File::with_name("config/config.json"))
        .add_source(config::Environment::with_prefix("RAILGUN"))
        .build()?
        .try_deserialize()?;
    let routers_data = serde_json::from_slice::<Vec<RouterData>>(&fs::read("import/Router.json")?)?;
    let mut routers: HashMap<u8, Router> = routers_data
        .iter()
        .map(|r| (r.id, Router::new(r, &config)))
        .collect();
    // let groups: Vec<GatewayGroup> = serde_json::from_slice(&fs::read("import/GatewayGroup.json")?)?;

    let mut server = Server::new(
        // &routers,
        // groups
        //     .iter()
        //     .map(|g| (g.id, g.routers(&groups, &routers_data)))
        //     .collect::<HashMap<u16, HashSet<u8>>>(),
    );

    let mut hello = Hello { time: 0 };

    let socket = UdpSocket::bind(SocketAddr::new(
        IpAddr::V6(Ipv6Addr::UNSPECIFIED),
        config.port,
    ))
    .await?;

    let mut timer = time::interval(INTERVAL);
    let mut buf = [0; 1500];

    loop {
        tokio::select! {
            Ok((len, src)) = socket.recv_from(&mut buf) => {
                if src == config.server {
                    // from server
                    let (message, _): (Change, usize) = bincode::decode_from_slice(&buf[..len], bincode::config::standard())?;
                    server.on_message(&socket, &message, &mut routers, &hello);
                    let report = Report {
                        id: config.id,
                        ack: server.ack,
                        peers: Vec::new()
                    };
                    let message = bincode::encode_to_vec(&report, bincode::config::standard())?;
                    let _ = socket.send_to(message.as_slice(), config.server);
                } else if let Some(peer) = Router::get(&mut routers, src){
                    // from client
                    let (message, _): (Hello, usize) = bincode::decode_from_slice(&buf[..len], bincode::config::standard())?;
                    peer.on_message(&message);
                }
            }

            _ = timer.tick() => {
                // to clients
                hello.time = SystemTime::now().duration_since(SystemTime::UNIX_EPOCH)?.as_millis() as u16;
                let message = bincode::encode_to_vec(&hello, bincode::config::standard())?;
                for peer in routers.values() {
                    let _ = socket.send_to(message.as_slice(), peer.link_address);
                }

                // to server
                let report = Report {
                    id: config.id,
                    ack: server.ack,
                    peers: routers
                        .values_mut()
                        .map(|peer| peer.update(hello.time))
                        .collect(),
                };
                let message = bincode::encode_to_vec(&report, bincode::config::standard())?;
                let _ = socket.send_to(message.as_slice(), config.server);
            }
        }
    }
}
