use crate::data::Router as RouterData;
use crate::protocol::{Hello, PeerQuality};
use crate::settings::{Settings, HISTORY, INTERVAL, TIMEOUT};
use average::Mean;
use std::collections::HashMap;
use std::net::{IpAddr, SocketAddr};
use std::time::SystemTime;

pub struct Router {
    pub link_address: SocketAddr,
    quality: PeerQuality,
    remote_time: u16,
    local_time: u16,
    history: Vec<Option<i16>>,
}

impl Router {
    pub fn get(routers: &mut HashMap<u8, Router>, link_address: SocketAddr) -> Option<&mut Router> {
        match link_address {
            SocketAddr::V4(addr) => {
                let id = addr.ip().octets()[3];
                routers.get_mut(&id)
            }
            SocketAddr::V6(_) => None,
        }
    }
    pub fn new(data: &RouterData, config: &Settings) -> Router {
        Router {
            link_address: SocketAddr::new(
                IpAddr::from([10, 200, data.id, config.id]),
                config.bind.port(),
            ),
            quality: PeerQuality {
                reliability: 0,
                jitter: 0,
                delay: 0,
            },
            local_time: 0,
            remote_time: 0,
            history: Vec::new(),
        }
    }

    pub fn reset(&mut self) {
        self.quality = PeerQuality {
            reliability: 0,
            jitter: 0,
            delay: 0,
        };
        self.history.clear();
    }

    pub fn on_message(&mut self, data: &Hello) {
        // 这个包发出距离上一个包
        let diff = data.time.wrapping_sub(self.remote_time) as i16;

        // 收到时间略小于或相等的，可能是网络乱序或重包，忽略
        if -(TIMEOUT.as_millis() as i16) < diff && diff <= 0 {
            return;
        }
        if 0 < diff && diff <= (TIMEOUT.as_millis() as i16) {
            // 差距较小，补上中间丢的包
            let step = (diff as f64 / INTERVAL.as_millis() as f64).round() as u8;
            for _ in 0..step - 1 {
                self.history.push(None);
            }
        } else {
            // 差距较大，就 reset
            self.reset();
        }

        self.remote_time = data.time;
        self.local_time = SystemTime::now()
            .duration_since(SystemTime::UNIX_EPOCH)
            .unwrap()
            .as_millis() as u16;

        let delay = self.local_time.wrapping_sub(self.remote_time) as i16;

        self.history.push(Some(delay));

        if self.history.len() > HISTORY as usize {
            self.history.drain(0..self.history.len() - HISTORY as usize);
        }

        let received: Vec<i16> = self.history.iter().filter_map(|&s| s).collect();
        assert!(!received.is_empty()); // 因为走到这里一定刚放过一个进去

        self.quality.reliability = (received.len() * 255 / HISTORY as usize) as u8;
        self.quality.delay = received
            .iter()
            .map(|&x| f64::from(x))
            .collect::<Mean>()
            .mean() as i16;
        self.quality.jitter = (1..received.len())
            .map(|i| f64::from(received[i].abs_diff(received[i - 1])))
            .collect::<Mean>()
            .mean() as u8;
    }

    pub(crate) fn update(&mut self, local_time: u16) -> PeerQuality {
        if self.quality.reliability > 0 {
            let diff = (local_time.wrapping_sub(self.local_time) as i16 as f64
                / INTERVAL.as_millis() as f64)
                .round() as i16;

            // 有几个包没到
            if diff > TIMEOUT.as_millis() as i16 {
                self.reset();
            } else if diff >= (INTERVAL.as_millis() * 2) as i16 {
                self.quality.reliability = self.quality.reliability.saturating_sub(255 / HISTORY);
            }
        }

        self.quality
    }
}
