use crate::connection::Connection;
use crate::data::Router as RouterData;
use crate::protocol::{Change, Hello, Report};
use crate::router::Router;
use crate::server::Server;
use crate::settings::{Settings, INTERVAL};
use config::Config;
use std::collections::HashMap;
use std::fs;
use std::time::SystemTime;
use tokio::net::UdpSocket;
use tokio::time;

mod connection;
mod data;
mod gateway_group;
mod protocol;
mod route_writer;
mod router;
mod server;
mod settings;

#[tokio::main]
async fn main() -> anyhow::Result<()> {
    let config: Settings = Config::builder().add_source(config::Environment::default()).build()?.try_deserialize()?;
    let routers_data = serde_json::from_slice::<Vec<RouterData>>(&fs::read("import/Router.json")?)?;
    let mut routers: HashMap<u8, Router> = routers_data.iter().map(|r| (r.id, Router::new(r, &config))).collect();
    let connections = serde_json::from_slice::<HashMap<u8, HashMap<u8, Connection>>>(&fs::read("import/connections.json")?)?;
    // let groups: Vec<GatewayGroup> = serde_json::from_slice(&fs::read("import/GatewayGroup.json")?)?;

    let mut server = Server::new(
        // &routers,
        // groups
        //     .iter()
        //     .map(|g| (g.id, g.routers(&groups, &routers_data)))
        //     .collect::<HashMap<u16, HashSet<u8>>>(),
    );

    let mut hello = Hello { time: 0 };

    let socket = UdpSocket::bind(config.bind).await?;

    let mut timer = time::interval(INTERVAL);
    let mut buf = [0; 1500];

    let mut syn: bool = true;

    loop {
        tokio::select! {
            biased; // 优先处理上面的

            result = socket.recv_from(&mut buf) => {
                let (len, src) = result?;
                if src == config.server {
                    // from server
                    let (message, _): (Change, usize) = bincode::decode_from_slice(&buf[..len], bincode::config::standard())?;
                    server.on_message(&message);
                    let report = Report {
                        id: config.id,
                        syn: message.rst,
                        ack: server.ack,
                        peers: Vec::new()
                    };
                    let message = bincode::encode_to_vec(&report, bincode::config::standard())?;
                    let _ = socket.send_to(message.as_slice(), config.server).await;
                    syn = false;
                } else if let Some(peer) = Router::get(&mut routers, src){
                    // from client
                    let (message, _): (Hello, usize) = bincode::decode_from_slice(&buf[..len], bincode::config::standard())?;
                    peer.on_message(&message);
                }
            }

            _ = timer.tick() => {
                // to clients
                hello.time = SystemTime::now().duration_since(SystemTime::UNIX_EPOCH)?.as_millis() as u16;
                let message = bincode::encode_to_vec(&hello, bincode::config::standard())?;
                for id in connections[&config.id].keys() {
                    let router = &routers[id];
                    let _ = socket.send_to(message.as_slice(), router.link_address).await;
                }

                // to server
                let report = Report {
                    id: config.id,
                    syn,
                    ack: server.ack,
                    peers: connections
                        .iter()
                        .filter(|(_, to)| to.contains_key(&config.id))
                        .map(|(from,_)|routers.get_mut(from).unwrap().update(hello.time))
                        .collect(),
                };
                let message = bincode::encode_to_vec(&report, bincode::config::standard())?;
                let _ = socket.send_to(message.as_slice(), config.server).await;
            }
        }
    }
}
