import { Socket } from 'dgram';

import config from '../config/config.json';

import { RouteWriter } from './RouteWriter';
import { Peer } from './Peer';
import { DownloadMessage, PeerMessage, PeerQuality, UploadMessage } from '../protocol';

export class Server {
  ack = 0;

  onMessage(socket: Socket, message: DownloadMessage, self: PeerMessage) {
    console.log(message);

    if (message.seq && this.ack !== message.seq) {
      console.log("seq mismatch rejected");
      return;
    }

    if (message.seq === 0) {
      RouteWriter.reset();
    }

    for (const [to, via] of Object.entries(message.via)) {
      RouteWriter.setVia(parseInt(to), via);
    }
    for (const [plan, to] of Object.entries(message.plan)) {
      RouteWriter.setVia(parseInt(plan), to);
    }
    RouteWriter.commit();
    this.ack = message.seq + 1;

    const response: UploadMessage = {
      id: self.id,
      ack: this.ack
    };

    socket.send(JSON.stringify(response), config.server_port, config.server_address);
  }

  update(socket: Socket, self: PeerMessage, peers: Peer[]) {
    const p: Record<number, PeerQuality> = {};

    for (const peer of peers) {
      if (peer.reliability === 0) {
        continue;
      }
      // 有几个包没到
      const step = Math.max(0, Math.floor((self.time - peer.time + peer.delay - config.interval / 2) / config.interval));
      if (step >= config.timeout) {
        peer.reset();
        continue;
      }

      const { id, delay } = peer;
      const reliability = (peer.reliability * (config.timeout - step)) / config.timeout;
      // TODO: jitter 没算
      p[id] = { delay: Math.round(delay), jitter: 0, reliability };
    }

    const message: UploadMessage = { id: self.id, ack: this.ack, peers: p };
    // console.log(message);
    socket.send(JSON.stringify(message), config.server_port, config.server_address);
  }
}
