import config from '../config/config.json';
import { PeerMessage, PeerQuality } from '../protocol';

export interface RouterConfig {
  id: number;
  address: string;
  subnets: string[];
}

export class Peer implements PeerMessage, PeerQuality, RouterConfig {
  id: number;
  address: string;
  subnets: string[];

  delay: number = 0;
  jitter: number = 0;
  reliability: number = 0;
  seq: number = 0;
  time: number = 0;

  constructor(config: RouterConfig) {
    Object.assign(this, config);
  }

  reset() {
    this.delay = 0;
    this.reliability = 0;
    this.seq = 0;
    this.time = 0;
  }

  onMessage(data: PeerMessage) {
    // console.log(data);
    if (data.seq === 0 || data.seq < this.seq - config.timeout || data.seq > this.seq + config.timeout) {
      // 收到 seq = 0 或 seq 与之前差距较大，就 reset
      this.reset();
    } else if (data.seq <= this.seq) {
      // 收到 seq 比已知略小的，忽略
      return;
    }
    // 全新或者 seq 比已知略大。

    const step = data.seq - this.seq;
    const delay = Date.now() - data.time;

    this.reliability = (this.reliability * (config.timeout - step)) / config.timeout + 1 / config.timeout;
    this.delay = (this.delay * (config.timeout - 1)) / config.timeout + delay / config.timeout;

    this.seq = data.seq;
    this.time = data.time;
  }

  update(time: number) {
    if (this.reliability === 0) {
      return;
    }
    // 有几个包没到
    const step = Math.floor((time - this.time + this.delay - config.interval) / config.interval);
    if (step > config.timeout) {
      this.reset();
    }
  }
}
