package https

import (
	"bytes"
	"crypto/rand"
	"crypto/rsa"
	"io"
	"os"
	"strings"
	"testing"
	"time"

	"github.com/miekg/coredns/server"
	"github.com/xenolf/lego/acme"
)

func TestUser(t *testing.T) {
	privateKey, err := rsa.GenerateKey(rand.Reader, 128)
	if err != nil {
		t.Fatalf("Could not generate test private key: %v", err)
	}
	u := User{
		Email:        "me@mine.com",
		Registration: new(acme.RegistrationResource),
		key:          privateKey,
	}

	if expected, actual := "me@mine.com", u.GetEmail(); actual != expected {
		t.Errorf("Expected email '%s' but got '%s'", expected, actual)
	}
	if u.GetRegistration() == nil {
		t.Error("Expected a registration resource, but got nil")
	}
	if expected, actual := privateKey, u.GetPrivateKey(); actual != expected {
		t.Errorf("Expected the private key at address %p but got one at %p instead ", expected, actual)
	}
}

func TestNewUser(t *testing.T) {
	email := "me@foobar.com"
	user, err := newUser(email)
	if err != nil {
		t.Fatalf("Error creating user: %v", err)
	}
	if user.key == nil {
		t.Error("Private key is nil")
	}
	if user.Email != email {
		t.Errorf("Expected email to be %s, but was %s", email, user.Email)
	}
	if user.Registration != nil {
		t.Error("New user already has a registration resource; it shouldn't")
	}
}

func TestSaveUser(t *testing.T) {
	storage = Storage("./testdata")
	defer os.RemoveAll(string(storage))

	email := "me@foobar.com"
	user, err := newUser(email)
	if err != nil {
		t.Fatalf("Error creating user: %v", err)
	}

	err = saveUser(user)
	if err != nil {
		t.Fatalf("Error saving user: %v", err)
	}
	_, err = os.Stat(storage.UserRegFile(email))
	if err != nil {
		t.Errorf("Cannot access user registration file, error: %v", err)
	}
	_, err = os.Stat(storage.UserKeyFile(email))
	if err != nil {
		t.Errorf("Cannot access user private key file, error: %v", err)
	}
}

func TestGetUserDoesNotAlreadyExist(t *testing.T) {
	storage = Storage("./testdata")
	defer os.RemoveAll(string(storage))

	user, err := getUser("user_does_not_exist@foobar.com")
	if err != nil {
		t.Fatalf("Error getting user: %v", err)
	}

	if user.key == nil {
		t.Error("Expected user to have a private key, but it was nil")
	}
}

func TestGetUserAlreadyExists(t *testing.T) {
	storage = Storage("./testdata")
	defer os.RemoveAll(string(storage))

	email := "me@foobar.com"

	// Set up test
	user, err := newUser(email)
	if err != nil {
		t.Fatalf("Error creating user: %v", err)
	}
	err = saveUser(user)
	if err != nil {
		t.Fatalf("Error saving user: %v", err)
	}

	// Expect to load user from disk
	user2, err := getUser(email)
	if err != nil {
		t.Fatalf("Error getting user: %v", err)
	}

	// Assert keys are the same
	if !PrivateKeysSame(user.key, user2.key) {
		t.Error("Expected private key to be the same after loading, but it wasn't")
	}

	// Assert emails are the same
	if user.Email != user2.Email {
		t.Errorf("Expected emails to be equal, but was '%s' before and '%s' after loading", user.Email, user2.Email)
	}
}

func TestGetEmail(t *testing.T) {
	// let's not clutter up the output
	origStdout := os.Stdout
	os.Stdout = nil
	defer func() { os.Stdout = origStdout }()

	storage = Storage("./testdata")
	defer os.RemoveAll(string(storage))
	DefaultEmail = "test2@foo.com"

	// Test1: Use email in config
	config := server.Config{
		TLS: server.TLSConfig{
			LetsEncryptEmail: "test1@foo.com",
		},
	}
	actual := getEmail(config, true)
	if actual != "test1@foo.com" {
		t.Errorf("Did not get correct email from config; expected '%s' but got '%s'", "test1@foo.com", actual)
	}

	// Test2: Use default email from flag (or user previously typing it)
	actual = getEmail(server.Config{}, true)
	if actual != DefaultEmail {
		t.Errorf("Did not get correct email from config; expected '%s' but got '%s'", DefaultEmail, actual)
	}

	// Test3: Get input from user
	DefaultEmail = ""
	stdin = new(bytes.Buffer)
	_, err := io.Copy(stdin, strings.NewReader("test3@foo.com\n"))
	if err != nil {
		t.Fatalf("Could not simulate user input, error: %v", err)
	}
	actual = getEmail(server.Config{}, true)
	if actual != "test3@foo.com" {
		t.Errorf("Did not get correct email from user input prompt; expected '%s' but got '%s'", "test3@foo.com", actual)
	}

	// Test4: Get most recent email from before
	DefaultEmail = ""
	for i, eml := range []string{
		"test4-3@foo.com",
		"test4-2@foo.com",
		"test4-1@foo.com",
	} {
		u, err := newUser(eml)
		if err != nil {
			t.Fatalf("Error creating user %d: %v", i, err)
		}
		err = saveUser(u)
		if err != nil {
			t.Fatalf("Error saving user %d: %v", i, err)
		}

		// Change modified time so they're all different, so the test becomes deterministic
		f, err := os.Stat(storage.User(eml))
		if err != nil {
			t.Fatalf("Could not access user folder for '%s': %v", eml, err)
		}
		chTime := f.ModTime().Add(-(time.Duration(i) * time.Second))
		if err := os.Chtimes(storage.User(eml), chTime, chTime); err != nil {
			t.Fatalf("Could not change user folder mod time for '%s': %v", eml, err)
		}
	}
	actual = getEmail(server.Config{}, true)
	if actual != "test4-3@foo.com" {
		t.Errorf("Did not get correct email from storage; expected '%s' but got '%s'", "test4-3@foo.com", actual)
	}
}
