﻿using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Collections.Generic;
using System.Text;
using System.Xml.Serialization;
using System.Globalization;

namespace DataEditorX.Core.Mse
{
    [XmlType(TypeName = "choice")]
    public class Choice
    {
        public Choice()
        {
            key = -1;
            _key="0";
            val = "";
            name = "";
        }
        [XmlAttribute("key")]
        public string _key;
        [XmlIgnoreAttribute()]
        private long key;
        [XmlIgnoreAttribute()]
        public long Key
        {
            set { key = value; }
            get {
                if (key >= 0)
                    return key;
                if (_key.StartsWith("0x"))
                    long.TryParse(_key.Replace("0x",""), NumberStyles.HexNumber, null, out key);
                else
                    long.TryParse(_key, out key);
                return key;
            }
        }
        [XmlAttribute("value")]
        public string val;
        [XmlAttribute("name")]
        public string name;
    }
    [XmlType(TypeName = "font")]
    public class MyFont
    {
        public MyFont()
        {
            fontStyle = FontStyle.Regular;
            fontName = "宋体";
            fontColor = "#000000";
            fontSize = 14.0f;
            font = null;
        }
        [XmlIgnoreAttribute()] 
        public Font font;
        [XmlAttribute("name")]
        public string fontName;
        [XmlAttribute("size")]
        public float fontSize;
        [XmlAttribute("color")]
        public string fontColor;
        [XmlAttribute("style")]
        public FontStyle fontStyle;

        public Font setSize(float size)
        {
            font = new Font(fontName, size, fontStyle);
            return font;
        }
    }
    [XmlType(TypeName = "Item")]
    public class Item
    {
        public Item()
        {
            font = new MyFont();
            name = "unknown";
            align = "left";
            src = "none";
            content = "";
            isMuilt = false;
            srcs = new List<Choice>();
            texts = new List<Choice>();
        }
        public bool isImage
        {
            get { return bmp != null; }
        }
        public bool isText
        {
            get { return !string.IsNullOrEmpty(content); }
        }
        public bool isLeft
        {
            get { return align.Equals("left"); }
        }
        public bool isRight
        {
            get { return align.Equals("right"); }
        }

        [XmlAttribute("name")]
        public string name;
        [XmlAttribute("left")]
        public float left;
        [XmlAttribute("top")]
        public float top;
        [XmlAttribute("right")]
        public float right;
        [XmlAttribute("bottom")]
        public float bottom;
        [XmlAttribute("width")]
        public float width;
        [XmlAttribute("height")]
        public float height;
        [XmlAttribute("align")]
        public string align;

        [XmlAttribute("image")]
        public string src;
        [XmlAttribute("text")]
        public string content;
        [XmlIgnoreAttribute()] 
        public string text;

        [XmlArray("Images")]
        [XmlArrayItem("choice")]
        public List<Choice> srcs;
        [XmlArray("Texts")]
        [XmlArrayItem("choice")]
        public List<Choice> texts; 

        [XmlIgnoreAttribute()] 
        public Bitmap bmp;
        [XmlElement("font")]
        public MyFont font;
        [XmlAttribute("visible")]
        public string visibleStr;
        [XmlIgnoreAttribute()]
        public bool visible;
        [XmlAttribute("multiline")]
        public bool isMuilt;
    }

    [XmlRoot("Group")]
    public class ItemGroup
    {
        public ItemGroup()
        {
            items = new List<Item>();
            name = "none";
        }
        [XmlElement("name")]
        public string name;
        [XmlElement("width")]
        public int width;
        [XmlElement("height")]
        public int height;

        [XmlArray("Items")]
        [XmlArrayItem("Item")]
        public List<Item> items;

        public Bitmap GetBitmap()
        {
            Bitmap bitmap = new Bitmap(width, height);
            Graphics graphics = Graphics.FromImage(bitmap);
            //合成：高质量，低速度
            graphics.CompositingQuality = CompositingQuality.HighQuality;
            //去除锯齿
            graphics.SmoothingMode = SmoothingMode.HighQuality;
            //偏移：高质量，低速度
            graphics.PixelOffsetMode = PixelOffsetMode.HighQuality;
            //插补算法
            graphics.InterpolationMode = InterpolationMode.HighQualityBicubic;

            foreach (Item item in items)
            {
                if (!item.visible)//不需要绘制
                    continue;
                //判断文字，图片
                if (item.isImage)
                {
                    //图片
                    DrawImage(graphics, item);
                }else
                {
                    //文字
                    DrawText(graphics, item);
                }
            }
            graphics.Dispose();
            return bitmap;
        }

        void DrawImage(Graphics graphics, Item item)
        {
            Bitmap bmp = item.bmp;
            RectangleF srcRect = new RectangleF(0, 0, bmp.Width, bmp.Height);
            RectangleF dstRect = new RectangleF(item.left, item.top, item.width, item.height);
            if (item.right >0)
            { //判断左右
                float left = width - bmp.Width - item.right;
                dstRect = new RectangleF(left, item.top, item.width, item.height);
            }
            graphics.DrawImage(bmp, dstRect, srcRect, GraphicsUnit.Pixel);
        }
        
        void DrawText(Graphics graphics, Item item)
        {
            string text = item.text;
            if (string.IsNullOrEmpty(text))
                return;
            Font font = item.font.font;
            float setp = font.Size / 5.0f;
            Color color = String2Color(item.font.fontColor);
            Brush bush = new SolidBrush(color);
            SizeF srcSize = new SizeF(item.width, item.height);
            RectangleF srcRect = new RectangleF(item.left, item.top, item.width, item.height);
            StringFormat strFormat = new StringFormat();
            strFormat.Alignment = StringAlignment.Near;//对齐
            if (item.isRight)
            {
                //右对齐
                strFormat.FormatFlags |= StringFormatFlags.DirectionRightToLeft;
            }
            if (item.isMuilt)
            {
                SizeF size = graphics.MeasureString(text, font, srcSize, strFormat);
                while (size.Height > srcSize.Height)//大小处理
                {
                    font = item.font.setSize(font.Size - setp);
                    size = graphics.MeasureString(text, font, srcSize, strFormat);
                }
            }
            else
            {
                //看到整行
                strFormat.FormatFlags |= StringFormatFlags.LineLimit;
                //不换行
                strFormat.FormatFlags |= StringFormatFlags.NoWrap;
                //未完成：字距
                SizeF size = graphics.MeasureString(text, font);
                srcRect = new RectangleF(item.left, item.top, item.width, item.height);
            }
            graphics.DrawString(text, font, bush, srcRect, strFormat);
        }
        static Color String2Color(string str)
        {
            //0123456
            //#ffffff
            if (str.Length < 7)
                return Color.Black;
            string r = str.Substring(1, 2);
            string g = str.Substring(3, 2);
            string b = str.Substring(5, 2);

            int ir, ig, ib;
            int.TryParse(r, NumberStyles.HexNumber, null, out ir);
            if (ir < 0 || ir > 0xff)
                ir = 0;
            int.TryParse(g, NumberStyles.HexNumber, null, out ig);
            if (ig < 0 || ig > 0xff)
                ig = 0;
            int.TryParse(b, NumberStyles.HexNumber, null, out ib);
            if (ib < 0 || ib > 0xff)
                ib = 0;
            return Color.FromArgb(ir, ig, ib);
        }
    }
}
