# PHPDTS 部署运维指南

## 系统要求

### 硬件要求
- **CPU**: 2核心以上 (推荐4核心)
- **内存**: 4GB以上 (推荐8GB)
- **存储**: 20GB以上可用空间
- **网络**: 100Mbps以上带宽

### 软件要求
- **操作系统**: Linux (Ubuntu 20.04+ / CentOS 8+)
- **Web服务器**: Apache 2.4+ 或 Nginx 1.18+
- **PHP**: 7.4+ (推荐8.0+)
- **数据库**: MySQL 5.7+ 或 MariaDB 10.3+

## 环境安装

### Ubuntu/Debian 环境
```bash
# 更新系统
sudo apt update && sudo apt upgrade -y

# 安装LAMP环境
sudo apt install apache2 mysql-server php php-mysql php-mbstring php-xml php-curl php-gd php-zip -y

# 启动服务
sudo systemctl enable apache2 mysql
sudo systemctl start apache2 mysql

# 配置MySQL安全设置
sudo mysql_secure_installation
```

### CentOS/RHEL 环境
```bash
# 安装EPEL仓库
sudo yum install epel-release -y

# 安装LAMP环境
sudo yum install httpd mariadb-server php php-mysql php-mbstring php-xml php-curl php-gd php-zip -y

# 启动服务
sudo systemctl enable httpd mariadb
sudo systemctl start httpd mariadb

# 配置MariaDB安全设置
sudo mysql_secure_installation
```

## 应用部署

### 1. 代码部署

#### 从Git仓库部署
```bash
# 克隆代码
cd /var/www/
sudo git clone https://github.com/your-repo/phpdts.git
sudo chown -R www-data:www-data phpdts/
sudo chmod -R 755 phpdts/
```

#### 手动上传部署
```bash
# 创建目录
sudo mkdir -p /var/www/phpdts
cd /var/www/phpdts

# 上传文件后设置权限
sudo chown -R www-data:www-data .
sudo chmod -R 755 .
sudo chmod -R 777 gamedata/
```

### 2. 数据库配置

#### 创建数据库
```sql
-- 登录MySQL
mysql -u root -p

-- 创建数据库
CREATE DATABASE phpdts CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

-- 创建用户
CREATE USER 'phpdts'@'localhost' IDENTIFIED BY 'your_secure_password';
GRANT ALL PRIVILEGES ON phpdts.* TO 'phpdts'@'localhost';
FLUSH PRIVILEGES;

-- 退出
EXIT;
```

#### 导入数据结构
```bash
# 导入基础数据结构
mysql -u phpdts -p phpdts < /var/www/phpdts/gamedata/sql/all.sql

# 导入初始数据（如果有）
mysql -u phpdts -p phpdts < /var/www/phpdts/install/initial_data.sql
```

### 3. 配置文件设置

#### 复制配置文件
```bash
cd /var/www/phpdts
cp config.inc.php.sample config.inc.php
```

#### 编辑配置文件
```php
// config.inc.php
<?php
// 数据库配置
$dbhost = 'localhost';
$dbuser = 'phpdts';
$dbpw = 'your_secure_password';
$dbname = 'phpdts';
$tablepre = 'bra_';
$gtablepre = 'acbra2_';

// 游戏配置
$gamecfg = 1;
$groomid = 1;

// 安全配置
$cookiedomain = '';
$cookiepath = '/';
$cookiepre = 'phpdts_';

// 调试模式（生产环境设为false）
define('DEBUG_MODE', false);
?>
```

### 4. Web服务器配置

#### Apache配置
```apache
# /etc/apache2/sites-available/phpdts.conf
<VirtualHost *:80>
    ServerName your-domain.com
    DocumentRoot /var/www/phpdts
    
    <Directory /var/www/phpdts>
        AllowOverride All
        Require all granted
        
        # 安全设置
        <Files "config.inc.php">
            Require all denied
        </Files>
        
        <FilesMatch "\.(sql|log|txt)$">
            Require all denied
        </FilesMatch>
    </Directory>
    
    # 日志配置
    ErrorLog ${APACHE_LOG_DIR}/phpdts_error.log
    CustomLog ${APACHE_LOG_DIR}/phpdts_access.log combined
</VirtualHost>
```

```bash
# 启用站点
sudo a2ensite phpdts.conf
sudo a2enmod rewrite
sudo systemctl reload apache2
```

#### Nginx配置
```nginx
# /etc/nginx/sites-available/phpdts
server {
    listen 80;
    server_name your-domain.com;
    root /var/www/phpdts;
    index index.php index.html;
    
    # 安全设置
    location ~ /config\.inc\.php$ {
        deny all;
    }
    
    location ~ \.(sql|log|txt)$ {
        deny all;
    }
    
    # PHP处理
    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.0-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
        include fastcgi_params;
    }
    
    # 静态文件缓存
    location ~* \.(jpg|jpeg|png|gif|ico|css|js)$ {
        expires 1y;
        add_header Cache-Control "public, immutable";
    }
}
```

```bash
# 启用站点
sudo ln -s /etc/nginx/sites-available/phpdts /etc/nginx/sites-enabled/
sudo nginx -t
sudo systemctl reload nginx
```

## SSL证书配置

### Let's Encrypt免费证书
```bash
# 安装Certbot
sudo apt install certbot python3-certbot-apache -y  # Apache
# 或
sudo apt install certbot python3-certbot-nginx -y   # Nginx

# 获取证书
sudo certbot --apache -d your-domain.com  # Apache
# 或
sudo certbot --nginx -d your-domain.com   # Nginx

# 自动续期
sudo crontab -e
# 添加以下行
0 12 * * * /usr/bin/certbot renew --quiet
```

## 性能优化

### 1. PHP优化

#### php.ini配置
```ini
# /etc/php/8.0/apache2/php.ini 或 /etc/php/8.0/fpm/php.ini

# 内存限制
memory_limit = 256M

# 执行时间
max_execution_time = 60

# 文件上传
upload_max_filesize = 10M
post_max_size = 10M

# OPcache配置
opcache.enable=1
opcache.memory_consumption=128
opcache.max_accelerated_files=4000
opcache.revalidate_freq=60
```

### 2. 数据库优化

#### MySQL配置
```ini
# /etc/mysql/mysql.conf.d/mysqld.cnf

[mysqld]
# 缓冲池大小（设为内存的70-80%）
innodb_buffer_pool_size = 2G

# 查询缓存
query_cache_type = 1
query_cache_size = 64M

# 连接数
max_connections = 200

# 慢查询日志
slow_query_log = 1
slow_query_log_file = /var/log/mysql/slow.log
long_query_time = 2
```

### 3. 缓存配置

#### Redis缓存（可选）
```bash
# 安装Redis
sudo apt install redis-server -y

# 配置Redis
sudo nano /etc/redis/redis.conf
# 设置密码
requirepass your_redis_password

# 重启Redis
sudo systemctl restart redis-server
```

## 监控和日志

### 1. 系统监控

#### 安装监控工具
```bash
# 安装htop和iotop
sudo apt install htop iotop -y

# 安装Netdata（可选）
bash <(curl -Ss https://my-netdata.io/kickstart.sh)
```

#### 监控脚本
```bash
#!/bin/bash
# monitor.sh - 系统监控脚本

# 检查磁盘使用率
DISK_USAGE=$(df -h / | awk 'NR==2 {print $5}' | sed 's/%//')
if [ $DISK_USAGE -gt 80 ]; then
    echo "警告：磁盘使用率超过80%"
fi

# 检查内存使用率
MEM_USAGE=$(free | grep Mem | awk '{printf("%.0f", $3/$2 * 100.0)}')
if [ $MEM_USAGE -gt 80 ]; then
    echo "警告：内存使用率超过80%"
fi

# 检查服务状态
systemctl is-active --quiet apache2 || echo "Apache服务异常"
systemctl is-active --quiet mysql || echo "MySQL服务异常"
```

### 2. 应用日志

#### 日志轮转配置
```bash
# /etc/logrotate.d/phpdts
/var/www/phpdts/gamedata/*.log {
    daily
    missingok
    rotate 30
    compress
    delaycompress
    notifempty
    copytruncate
}
```

#### 错误监控
```php
// 在config.inc.php中添加
// 错误日志配置
ini_set('log_errors', 1);
ini_set('error_log', '/var/www/phpdts/gamedata/php_errors.log');

// 自定义错误处理
function custom_error_handler($errno, $errstr, $errfile, $errline) {
    $error_message = date('Y-m-d H:i:s') . " - Error: $errstr in $errfile on line $errline\n";
    error_log($error_message, 3, '/var/www/phpdts/gamedata/app_errors.log');
}
set_error_handler('custom_error_handler');
```

## 备份策略

### 1. 数据库备份

#### 自动备份脚本
```bash
#!/bin/bash
# backup_db.sh - 数据库备份脚本

DB_NAME="phpdts"
DB_USER="phpdts"
DB_PASS="your_password"
BACKUP_DIR="/backup/mysql"
DATE=$(date +%Y%m%d_%H%M%S)

# 创建备份目录
mkdir -p $BACKUP_DIR

# 执行备份
mysqldump -u$DB_USER -p$DB_PASS $DB_NAME > $BACKUP_DIR/phpdts_$DATE.sql

# 压缩备份文件
gzip $BACKUP_DIR/phpdts_$DATE.sql

# 删除7天前的备份
find $BACKUP_DIR -name "phpdts_*.sql.gz" -mtime +7 -delete

echo "数据库备份完成: phpdts_$DATE.sql.gz"
```

#### 定时备份
```bash
# 添加到crontab
sudo crontab -e

# 每天凌晨2点备份
0 2 * * * /path/to/backup_db.sh

# 每小时增量备份（可选）
0 * * * * /path/to/incremental_backup.sh
```

### 2. 文件备份

#### 代码备份
```bash
#!/bin/bash
# backup_files.sh - 文件备份脚本

SOURCE_DIR="/var/www/phpdts"
BACKUP_DIR="/backup/files"
DATE=$(date +%Y%m%d_%H%M%S)

# 创建备份
tar -czf $BACKUP_DIR/phpdts_files_$DATE.tar.gz -C /var/www phpdts

# 删除30天前的备份
find $BACKUP_DIR -name "phpdts_files_*.tar.gz" -mtime +30 -delete

echo "文件备份完成: phpdts_files_$DATE.tar.gz"
```

## 安全加固

### 1. 系统安全

#### 防火墙配置
```bash
# 安装UFW
sudo apt install ufw -y

# 配置防火墙规则
sudo ufw default deny incoming
sudo ufw default allow outgoing
sudo ufw allow ssh
sudo ufw allow 80/tcp
sudo ufw allow 443/tcp

# 启用防火墙
sudo ufw enable
```

#### 系统更新
```bash
# 自动安全更新
sudo apt install unattended-upgrades -y
sudo dpkg-reconfigure -plow unattended-upgrades
```

### 2. 应用安全

#### 文件权限
```bash
# 设置正确的文件权限
sudo chown -R www-data:www-data /var/www/phpdts
sudo find /var/www/phpdts -type d -exec chmod 755 {} \;
sudo find /var/www/phpdts -type f -exec chmod 644 {} \;
sudo chmod 777 /var/www/phpdts/gamedata/
```

#### 敏感文件保护
```apache
# .htaccess文件保护
<Files "config.inc.php">
    Order allow,deny
    Deny from all
</Files>

<FilesMatch "\.(sql|log|txt)$">
    Order allow,deny
    Deny from all
</FilesMatch>
```

## 故障排除

### 常见问题

#### 1. 数据库连接失败
```bash
# 检查MySQL服务状态
sudo systemctl status mysql

# 检查连接权限
mysql -u phpdts -p -e "SHOW GRANTS;"

# 检查配置文件
grep -E "(dbhost|dbuser|dbname)" /var/www/phpdts/config.inc.php
```

#### 2. 权限问题
```bash
# 检查文件权限
ls -la /var/www/phpdts/gamedata/

# 重新设置权限
sudo chown -R www-data:www-data /var/www/phpdts/gamedata/
sudo chmod -R 777 /var/www/phpdts/gamedata/
```

#### 3. 性能问题
```bash
# 检查系统资源
htop
iotop
df -h

# 检查MySQL进程
mysqladmin -u root -p processlist

# 检查慢查询
sudo tail -f /var/log/mysql/slow.log
```

### 应急处理

#### 服务重启
```bash
# 重启Web服务器
sudo systemctl restart apache2  # 或 nginx

# 重启数据库
sudo systemctl restart mysql

# 重启PHP-FPM（如果使用）
sudo systemctl restart php8.0-fpm
```

#### 快速恢复
```bash
# 从备份恢复数据库
gunzip -c /backup/mysql/phpdts_20231201_020000.sql.gz | mysql -u phpdts -p phpdts

# 从备份恢复文件
tar -xzf /backup/files/phpdts_files_20231201_020000.tar.gz -C /var/www/
```

---

*本指南提供了PHPDTS的完整部署和运维方案，确保系统稳定可靠运行。*
