# PHPDTS 开发指南

## 开发环境搭建

### 系统要求
- **PHP**: 7.4 或更高版本
- **MySQL**: 5.7 或更高版本
- **Web服务器**: Apache 2.4+ 或 Nginx 1.18+
- **操作系统**: Linux (推荐) / Windows / macOS

### 开发工具推荐
- **IDE**: PhpStorm / VS Code / Sublime Text
- **数据库管理**: phpMyAdmin / MySQL Workbench
- **版本控制**: Git
- **调试工具**: Xdebug

### 本地环境配置

#### 1. 安装LAMP/WAMP环境
```bash
# Ubuntu/Debian
sudo apt update
sudo apt install apache2 mysql-server php php-mysql php-mbstring

# CentOS/RHEL
sudo yum install httpd mariadb-server php php-mysql php-mbstring
```

#### 2. 配置虚拟主机
```apache
<VirtualHost *:80>
    ServerName phpdts.local
    DocumentRoot /var/www/phpdts
    <Directory /var/www/phpdts>
        AllowOverride All
        Require all granted
    </Directory>
</VirtualHost>
```

#### 3. 数据库初始化
```sql
CREATE DATABASE phpdts_dev CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
CREATE USER 'phpdts'@'localhost' IDENTIFIED BY 'password';
GRANT ALL PRIVILEGES ON phpdts_dev.* TO 'phpdts'@'localhost';
FLUSH PRIVILEGES;
```

## 项目结构详解

### 核心目录说明
```
phpdts/
├── include/                  # 核心功能库
│   ├── common.inc.php        # 公共初始化文件
│   ├── game/                 # 游戏逻辑模块
│   │   ├── battle.func.php   # 战斗系统
│   │   ├── item.*.php        # 物品系统模块
│   │   ├── club*.func.php    # 社团系统
│   │   └── ...
│   ├── admin/                # 管理功能
│   └── ...
├── gamedata/                 # 游戏数据
│   ├── cache/                # 配置缓存
│   ├── sql/                  # 数据库结构
│   └── ...
├── templates/                # 模板文件
│   └── default/              # 默认模板
└── ...
```

### 文件命名规范
- **功能文件**: `功能名.func.php` (如：search.func.php, battle.func.php)
- **类文件**: `类名.class.php` (如：db_mysqli.class.php)
- **配置文件**: `配置名_版本.php` (如：gamecfg_1.php, resources_1.php)
- **模板文件**: `模板名.htm` (如：game.htm, battle.htm)
- **SQL文件**: `表名.sql` (如：players.sql, all.sql)
- **特殊配置**: 无版本号的配置文件 (如：club22cfg.php, fishing.php)

### 模板系统架构
```php
// 模板加载机制
function template($tplname) {
    global $templateid;
    return GAME_ROOT."./templates/{$templateid}/{$tplname}.htm";
}

// 模板解析和缓存
function parse_template($file, $templateid, $tpldir) {
    // 解析模板语法，生成PHP文件到 gamedata/templates/
    // 支持变量替换、条件语句、循环等
}
```

#### 模板文件结构
- `templates/default/` - 默认模板源文件
- `gamedata/templates/` - 编译后的模板缓存
- 支持多模板切换（通过u_templateid）
- 模板变量自动提取（extract函数）
- 支持模板语法：{变量名}、{if 条件}、{template 子模板}、{eval PHP代码}

## 编码规范

### PHP编码标准

#### 1. 基本格式
```php
<?php
// 文件头部注释
if(!defined('IN_GAME')) {
    exit('Access Denied');
}

// 函数定义
function function_name($param1, $param2 = null) {
    global $global_var;
    
    // 函数体
    if ($condition) {
        // 代码块
    }
    
    return $result;
}
```

#### 2. 变量命名
```php
// 变量使用小写字母和下划线
$user_name = 'example';
$player_data = array();

// 常量使用大写字母
define('GAME_ROOT', './');
define('MAX_PLAYERS', 100);

// 全局变量使用描述性名称
global $db, $tablepre, $pdata;
```

#### 3. 数组和对象
```php
// 数组定义
$config = array(
    'database' => array(
        'host' => 'localhost',
        'name' => 'phpdts'
    ),
    'game' => array(
        'max_players' => 100
    )
);

// 关联数组访问
$host = $config['database']['host'];
```

### 数据库操作规范

#### 1. 查询构建
```php
// 使用预处理语句
$stmt = $db->prepare("SELECT * FROM {$tablepre}players WHERE name = ? AND type = ?");
$stmt->bind_param('si', $player_name, $player_type);
$stmt->execute();
$result = $stmt->get_result();

// 避免直接拼接SQL
// 错误示例：$sql = "SELECT * FROM players WHERE name = '$name'";
```

#### 2. 数据验证
```php
// 输入验证
function validate_player_name($name) {
    if (empty($name) || strlen($name) > 40) {
        return false;
    }
    if (!preg_match('/^[a-zA-Z0-9_\x{4e00}-\x{9fa5}]+$/u', $name)) {
        return false;
    }
    return true;
}

// 数据清理
$name = trim($name);
$name = htmlspecialchars($name, ENT_QUOTES, 'UTF-8');
```

## 功能开发流程

### 1. 新功能开发步骤

#### 步骤1: 需求分析
- 明确功能需求和目标
- 分析对现有系统的影响
- 设计数据结构和接口

#### 步骤2: 数据库设计
```sql
-- 创建新表或修改现有表
ALTER TABLE {tablepre}players ADD COLUMN new_feature_data TEXT;

-- 创建索引
CREATE INDEX idx_new_feature ON {tablepre}players(new_feature_id);
```

#### 步骤3: 后端逻辑实现
```php
// 在 include/game/ 下创建功能文件
// include/game/new_feature.func.php

if(!defined('IN_GAME')) {
    exit('Access Denied');
}

function new_feature_process(&$data) {
    global $db, $tablepre, $log;
    
    // 功能逻辑实现
    // ...
    
    return $result;
}
```

#### 步骤4: 前端界面开发
```html
<!-- templates/default/new_feature.htm -->
<div class="feature-container">
    <h3>新功能</h3>
    <form method="post" action="game.php">
        <input type="hidden" name="mode" value="new_feature">
        <input type="submit" value="执行功能">
    </form>
</div>
```

#### 步骤5: 集成测试
- 单元测试
- 集成测试
- 性能测试

### 2. 物品系统扩展示例

#### 创建新物品类型
```php
// include/game/item.new_type.php
if(!defined('IN_GAME')) {
    exit('Access Denied');
}

function itemuse_new_type($itmk, $itms, $itmsk, $itme, $itmnum, &$data) {
    global $log, $mode;
    extract($data, EXTR_REFS);
    
    // 物品使用逻辑
    switch($itmk) {
        case 'NT001':
            // 特定物品逻辑
            $log .= '使用了新类型物品！<br>';
            break;
        default:
            $log .= '未知的新类型物品。<br>';
    }
    
    $mode = 'command';
}
```

#### 注册物品类型
```php
// 在 item.main.php 中添加
include_once GAME_ROOT.'./include/game/item.new_type.php';

// 在物品使用逻辑中添加分支
if(substr($itmk, 0, 2) == 'NT') {
    itemuse_new_type($itmk, $itms, $itmsk, $itme, $itmnum, $data);
    return;
}
```

## 调试和测试

### 1. 调试技巧

#### 日志调试
```php
// 添加调试信息
$log .= "<!-- DEBUG: 变量值 = {$variable} -->";

// 使用错误日志
error_log("调试信息: " . print_r($data, true));
```

#### 开发工具
```php
// devtools.php 中的调试功能
if(defined('DEBUG_MODE') && DEBUG_MODE) {
    // 显示详细错误信息
    ini_set('display_errors', 1);
    error_reporting(E_ALL);
}
```

### 2. 测试策略

#### 单元测试
```php
// 简单的测试函数
function test_function($input, $expected) {
    $result = target_function($input);
    if($result === $expected) {
        echo "测试通过\n";
    } else {
        echo "测试失败: 期望 {$expected}, 得到 {$result}\n";
    }
}
```

#### 集成测试
- 测试完整的用户流程
- 验证数据库操作正确性
- 检查权限和安全性

## 性能优化

### 1. 数据库优化

#### 查询优化
```php
// 避免N+1查询问题
// 错误示例：
foreach($players as $player) {
    $items = get_player_items($player['pid']); // 每次都查询数据库
}

// 正确示例：
$player_ids = array_column($players, 'pid');
$all_items = get_players_items($player_ids); // 一次查询获取所有数据
```

#### 索引使用
```sql
-- 为常用查询添加索引
CREATE INDEX idx_player_location ON players(pls, type);
CREATE INDEX idx_chat_time ON chat(chattime);
```

### 2. 代码优化

#### 缓存机制
```php
// 配置文件缓存
function load_config($name) {
    static $cache = array();
    
    if(!isset($cache[$name])) {
        $cache[$name] = include "config/{$name}.php";
    }
    
    return $cache[$name];
}
```

#### 内存管理
```php
// 及时释放大变量
unset($large_array);

// 避免内存泄漏
$db->free_result($result);
```

## 安全开发

### 1. 输入验证
```php
// 验证用户输入
function validate_input($input, $type) {
    switch($type) {
        case 'int':
            return filter_var($input, FILTER_VALIDATE_INT);
        case 'string':
            return filter_var($input, FILTER_SANITIZE_STRING);
        case 'email':
            return filter_var($input, FILTER_VALIDATE_EMAIL);
    }
    return false;
}
```

### 2. 权限检查
```php
// 检查用户权限
function check_permission($user, $action) {
    global $admin_cmd_list;
    
    if(!isset($admin_cmd_list[$action])) {
        return false;
    }
    
    return $user['groupid'] >= $admin_cmd_list[$action];
}
```

### 3. 数据加密
```php
// 密码加密
function hash_password($password) {
    return md5($password); // 注意：实际项目中建议使用更安全的哈希算法
}

// 敏感数据处理
function encrypt_sensitive_data($data) {
    // 实现加密逻辑
    return base64_encode($data);
}
```

## 版本控制

### Git工作流程
```bash
# 创建功能分支
git checkout -b feature/new-feature

# 提交更改
git add .
git commit -m "feat: 添加新功能"

# 合并到主分支
git checkout main
git merge feature/new-feature

# 删除功能分支
git branch -d feature/new-feature
```

### 提交信息规范
- `feat:` 新功能
- `fix:` 修复bug
- `docs:` 文档更新
- `style:` 代码格式调整
- `refactor:` 代码重构
- `test:` 测试相关
- `chore:` 构建过程或辅助工具的变动

## 部署指南

### 生产环境部署
1. **代码部署**: 使用Git或FTP上传代码
2. **配置更新**: 修改生产环境配置
3. **数据库迁移**: 执行数据库更新脚本
4. **权限设置**: 确保目录权限正确
5. **服务重启**: 重启Web服务器

### 监控和维护
- 定期备份数据库
- 监控系统性能
- 检查错误日志
- 更新安全补丁

---

*本指南为PHPDTS开发提供了全面的技术指导，帮助开发者快速上手并遵循最佳实践。*
