import { Injectable, Logger } from '@nestjs/common';
import moment from 'moment';
import PQueue from 'p-queue';
import _ from 'lodash';
import qs from 'qs';
import cookie from 'cookie';
import { ConfigService } from '@nestjs/config';
import { HttpService } from '@nestjs/axios';

const blacklistedWords = ['曲', '词', '曲', '詞', '[Vv]ocal', ':', '：'];
const blacklistedRegex = blacklistedWords.map((word) => new RegExp(word));

interface LoginInfo {
  email: string;
  password: string;
}

@Injectable()
export class AppService extends Logger {
  cache: string[];
  lastFetchTime: moment.Moment;
  queue = new PQueue({ concurrency: 1 });
  loginInfo: LoginInfo;
  constructor(config: ConfigService, private http: HttpService) {
    super('ygopro-tips-biu-generator');
    if (config.get<string>('BIU_EMAIL') && config.get<string>('BIU_PASSWORD')) {
      this.loginInfo = {
        email: config.get<string>('BIU_EMAIL'),
        password: config.get<string>('BIU_PASSWORD'),
      };
    }
  }

  async getTips(): Promise<string[]> {
    return this.fetchTips();
  }

  async fetchTips(): Promise<string[]> {
    return await this.queue.add(async () => this.fetchTipsTask());
  }

  async getLyricsOfSong(id: number): Promise<string[]> {
    try {
      this.log(`Fetching lyrics of song ${id}.`);
      const { data } = await this.http
        .get(`https://web.biu.moe/s${id}`, {
          responseType: 'document',
          timeout: 30000,
        })
        .toPromise();
      const lyricMatches = (data as string).match(
        /\[\d+:\d+[:\.]\d+\]([^<]+)/g,
      );
      if (!lyricMatches) {
        this.log(`Song ${id} has no lyrics.`);
        return [];
      }
      const lyrics = lyricMatches
        .map((m) =>
          decodeURIComponent(m.match(/\[\d+:\d+[:\.]\d+\]([^<]+)/)[1].trim()),
        )
        .filter((lyric) =>
          blacklistedRegex.every((regex) => !lyric.match(regex)),
        );
      this.log(`${lyrics.length} lines of lyrics found in song ${id}.`);
      return lyrics;
    } catch (e) {
      this.error(`Failed fetching lyrics of ${id}: ${e.toString()}`);
      return [];
    }
  }

  async getLoginCookie() {
    if (!this.loginInfo) {
      return undefined;
    }
    try {
      this.log(`Logging in.`);
      const ret = await this.http
        .post(
          'https://web.biu.moe/User/doLogin',
          qs.stringify(this.loginInfo),
          {
            responseType: 'json',
            timeout: 30000,
          },
        )
        .toPromise();
      if (!ret.data.status) {
        this.error(`Login failed: ${ret.data.message}`);
        return undefined;
      }
      const setCookieContent = ret.headers['set-cookie'] as string[];
      if (!setCookieContent) {
        return undefined;
      }
      const contentHits = setCookieContent.find((c) => c.includes('biuInfo'));
      if (!contentHits) {
        return undefined;
      }
      const token = cookie.parse(contentHits).biuInfo as string;
      this.log(`Login success: ${token}`);
      return cookie.serialize('biuInfo', token);
    } catch (e) {
      this.error(`Login error: ${e.toString()}`);
      return undefined;
    }
  }

  async fetchTipsTask(): Promise<string[]> {
    if (
      this.cache &&
      this.lastFetchTime &&
      moment().diff(this.lastFetchTime) <= 3600000
    ) {
      return this.cache;
    }
    try {
      this.log(`Fetching song list.`);
      const { data } = await this.http
        .get('https://web.biu.moe/Index/home', {
          responseType: 'document',
          headers: {
            Cookie: (await this.getLoginCookie()) || '',
          },
          timeout: 30000,
        })
        .toPromise();
      const urls = (data as string).match(/href="\/s(\d+)"/g);
      if (!urls) {
        this.error(`No songs found, retrying.`);
        return await this.fetchTipsTask();
      }
      this.log(`${urls.length} songs found.`);
      const songIds = urls.map((m) => parseInt(m.match(/href="\/s(\d+)"/)[1]));
      const lyrics = _.flatten(
        await Promise.all(songIds.map((id) => this.getLyricsOfSong(id))),
      );
      this.log(`${lyrics.length} lines of lyrics got.`);
      if (lyrics.length) {
        this.cache = lyrics;
        this.lastFetchTime = moment();
        return lyrics;
      } else {
        this.log(`No lyrics found, retrying.`);
        return await this.fetchTipsTask();
      }
    } catch (e) {
      this.error(`Errored fetching song list, retrying: ${e.toString()}`);
      return await this.fetchTipsTask();
    }
  }
}
