import lzma from 'lzma-native';
import _ from 'lodash';
import fs from 'fs';

export class replayHeader {
  public static replayCompressedFlag = 0x1;

  public static replayTagFlag = 0x2;

  public static replayDecodedFlag = 0x4;

  id = 0;
  version = 0;
  flag = 0;
  seed = 0;
  dataSizeRaw: number[] = [];
  hash = 0;
  props: number[] = [];

  public getDataSize() {
    return (
      this.dataSizeRaw[0] +
      this.dataSizeRaw[1] * 0x100 +
      this.dataSizeRaw[2] * 0x10000 +
      this.dataSizeRaw[3] * 0x1000000
    );
  }

  public getIsTag() {
    return (this.flag & replayHeader.replayTagFlag) > 0;
  }

  public getIsCompressed() {
    return (this.flag & replayHeader.replayCompressedFlag) > 0;
  }

  public getLzmaHeader() {
    const bytes = [].concat(
      this.props.slice(0, 5),
      this.dataSizeRaw,
      [0, 0, 0, 0],
    );
    return Buffer.from(bytes);
  }

  public get dataSize() {
    return this.getDataSize();
  }

  public get isTag() {
    return this.getIsTag();
  }

  public get isCompressed() {
    return this.getIsCompressed();
  }
}

export class ReplayReader {
  pointer = 0;

  constructor(public buffer: Buffer) {}

  public readByte() {
    const answer = this.buffer.readUInt8(this.pointer);
    this.pointer += 1;
    return answer;
  }

  public readByteArray(length: number) {
    const answer: number[] = [];
    _.range(1, length + 1).forEach((i) => answer.push(this.readByte()));
    return answer;
  }

  public readInt8() {
    const answer = this.buffer.readInt8(this.pointer);
    this.pointer += 1;
    return answer;
  }

  public readUInt8() {
    const answer = this.buffer.readUInt8(this.pointer);
    this.pointer += 1;
    return answer;
  }

  public readInt16() {
    const answer = this.buffer.readInt16LE(this.pointer);
    this.pointer += 2;
    return answer;
  }

  public readInt32() {
    const answer = this.buffer.readInt32LE(this.pointer);
    this.pointer += 4;
    return answer;
  }

  public readAll() {
    const answer = this.buffer.slice(this.pointer);
    // @pointer = 0
    return answer;
  }

  public readString(length: number) {
    if (this.pointer + length > this.buffer.length) {
      return null;
    }
    const full = this.buffer
      .slice(this.pointer, this.pointer + length)
      .toString('utf16le');
    const answer = full.split('\u0000')[0];
    this.pointer += length;
    return answer;
  }

  public readRaw(length: number) {
    if (this.pointer + length > this.buffer.length) {
      return null;
    }
    const answer = this.buffer.slice(this.pointer, this.pointer + length);
    this.pointer += length;
    return answer;
  }
}

export class Deck {
  main: number[];
  ex: number[];
}

async function lzmaDecompress(buf: Buffer): Promise<Buffer> {
  return new Promise((resolve) => {
    lzma.decompress(buf, 5, resolve);
  });
}

export class Replay {
  header: replayHeader;
  hostName = '';
  clientName = '';
  startLp = 0;
  startHand = 0;
  drawCount = 0;
  opt = 0;
  hostDeck: Deck;
  clientDeck: Deck;

  tagHostName = '';
  tagClientName = '';
  tagHostDeck: Deck;
  tagClientDeck: Deck;

  responses: Buffer[];
  constructor() {
    this.header = null;
    this.hostName = '';
    this.clientName = '';
    this.startLp = 0;
    this.startHand = 0;
    this.drawCount = 0;
    this.opt = 0;
    this.hostDeck = null;
    this.clientDeck = null;

    this.tagHostName = null;
    this.tagClientName = null;
    this.tagHostDeck = null;
    this.tagClientDeck = null;

    this.responses = null;
  }

  public getDecks() {
    if (this.isTag) {
      return [
        this.hostDeck,
        this.clientDeck,
        this.tagHostDeck,
        this.tagClientDeck,
      ];
    } else {
      return [this.hostDeck, this.clientDeck];
    }
  }

  public getIsTag() {
    return this.header && this.header.isTag;
  }

  public static async fromFile(filePath: string) {
    return Replay.fromBuffer(await fs.promises.readFile(filePath));
  }

  public static async fromBuffer(buffer: Buffer) {
    let reader = new ReplayReader(buffer);
    const header = Replay.readHeader(reader);
    const restBuffer = reader.readAll();
    const lzmaBuffer = Buffer.concat([header.getLzmaHeader(), restBuffer]);
    let decompressed: Buffer;
    if (!header.isCompressed) {
      // console.error(`no decompress`);
      decompressed = restBuffer;
    } else {
      // console.error(`decompressing`);
      decompressed = Buffer.from(await lzmaDecompress(lzmaBuffer));
      // console.error(`decompressed`);
    }
    reader = new ReplayReader(decompressed);
    const replay = Replay.readReplay(header, reader);
    return replay;
  }

  public static readHeader(reader: ReplayReader) {
    const header = new replayHeader();
    header.id = reader.readInt32();
    header.version = reader.readInt32();
    header.flag = reader.readInt32();
    header.seed = reader.readInt32();
    header.dataSizeRaw = reader.readByteArray(4);
    header.hash = reader.readInt32();
    header.props = reader.readByteArray(8);
    return header;
  }

  public static readReplay(header: replayHeader, reader: ReplayReader) {
    const replay = new Replay();
    // console.error('reading info');
    replay.header = header;
    replay.hostName = reader.readString(40);
    if (header.isTag) {
      replay.tagHostName = reader.readString(40);
    }
    if (header.isTag) {
      replay.tagClientName = reader.readString(40);
    }
    replay.clientName = reader.readString(40);
    replay.startLp = reader.readInt32();
    replay.startHand = reader.readInt32();
    replay.drawCount = reader.readInt32();
    replay.opt = reader.readInt32();
    // console.error(JSON.stringify(replay, null, 2));
    // console.error('reading deck');
    replay.hostDeck = Replay.readDeck(reader);
    if (header.isTag) {
      replay.tagHostDeck = Replay.readDeck(reader);
    }
    if (header.isTag) {
      replay.tagClientDeck = Replay.readDeck(reader);
    }
    replay.clientDeck = Replay.readDeck(reader);
    // console.error('reading response');
    replay.responses = Replay.readResponses(reader);
    return replay;
  }

  public static readDeck(reader: ReplayReader) {
    const deck = new Deck();
    deck.main = Replay.readDeckPack(reader);
    deck.ex = Replay.readDeckPack(reader);
    return deck;
  }

  public static readDeckPack(reader: ReplayReader) {
    const length = reader.readInt32();
    if (length > 128) {
      return [];
    }
    const answer: number[] = [];
    _.range(1, length + 1).forEach(() => answer.push(reader.readInt32()));
    return answer;
  }

  public static readResponses(reader: ReplayReader) {
    let length: number;
    let single: Buffer;
    const answer: Buffer[] = [];
    while (true) {
      try {
        length = reader.readUInt8();
        if (length > 64) {
          length = 64;
        }
        single = reader.readRaw(length);
        if (!single) {
          break;
        }
        answer.push(single);
      } catch (_error) {
        break;
      }
    }
    return answer;
  }

  public get decks() {
    return this.getDecks();
  }

  public get isTag() {
    return this.getIsTag();
  }
}
