import path from 'node:path';
import PQueue from 'p-queue';
import * as fs from 'node:fs';
import axios from 'axios';
import { Minify } from './minifier/luamin';

export interface YGOProLuaMinifyOptions {
  root: string;
  constantContent?: string;
  constantPath?: string;
  constantUrl?: string;
}
export class YGOProLuaMinify {
  constructor(private options: YGOProLuaMinifyOptions) {}

  constantContent = this.options.constantContent || '';

  constantQueue = new PQueue({ concurrency: 1 });

  async getConstantContent(): Promise<string> {
    if (this.constantContent) return this.constantContent;

    const resolveContent = (content: string) => {
      this.constantContent = content;
      return content;
    };

    return this.constantQueue.add(async () => {
      if (this.constantContent) return this.constantContent;

      const constantPath = path.resolve(
        this.options.root,
        this.options.constantPath || 'constant.lua',
      );

      // if file exists, read it
      try {
        const content = await fs.promises.readFile(constantPath, 'utf-8');
        console.log(`Loaded constant.lua from ${constantPath}`);
        return resolveContent(content);
      } catch (e) {
        // ignore
      }

      const url =
        this.options.constantUrl ||
        'https://cdntx.moecube.com/koishipro/content/script/constant.lua';

      const res = await axios.get<string>(url, {
        responseType: 'text',
      });

      console.log(`Loaded constant.lua from ${url}`);

      return resolveContent(res.data);
    });
  }

  constants: { key: string; value: string; valueNum: number }[] = [];

  parseConstants(content: string) {
    const constantsMap = new Map<
      string,
      { key: string; value: string; valueNum: number }
    >();
    for (const line of content.split('\n')) {
      const withoutComment = line.split('--')[0];
      const [keyPart, valuePart] = withoutComment.split('=');
      const key = keyPart?.trim();
      const value = valuePart?.trim();
      if (!key || !value) continue;
      const valueParts = value.split(/[+|]/).map((v) => {
        if (/[A-Z_]/.test(v)) {
          // reference to another constant
          const ref = constantsMap.get(v.trim());
          if (ref) return ref.valueNum;
          return 0;
        }
        return parseInt(v.trim()) || 0;
      });
      const valueNum = valueParts.reduce((a, b) => a | b, 0);
      const valueDec = valueNum.toString(10);
      // const valueHex = '0x' + valueNum.toString(16).toLowerCase();
      const pickValue = valueDec; // valueHex.length < valueDec.length ? valueHex : valueDec;
      constantsMap.set(key, { key, value: pickValue, valueNum });
      if (key.endsWith('_DEFENSE')) {
        // add _DEFENCE version too
        const altKey = key.replace(/_DEFENSE$/, '_DEFENCE');
        constantsMap.set(altKey, { key: altKey, value: pickValue, valueNum });
      }
    }
    // sort with key length desc
    this.constants = Array.from(constantsMap.values()).filter(
      (c) => c.key.length > c.value.length,
    );
    this.constants.sort((a, b) => b.key.length - a.key.length);
    console.log(`Parsed ${this.constants.length} constants`);
  }

  async loadConstants() {
    const content = await this.getConstantContent();
    this.parseConstants(content);
  }

  async minifyFile(filePath: string) {
    const absPath = path.resolve(this.options.root, filePath);
    let content = await fs.promises.readFile(absPath, 'utf-8');
    if (!filePath.endsWith('constant.lua')) {
      const isUtility = filePath.endsWith('utility.lua');
      const contentLinesPre = content
        .split('\n')
        .map((line) => line.trim())
        .filter((line) => line.length > 0);
      for (let i = 0; i < contentLinesPre.length; i++) {
        if (isUtility && contentLinesPre[i].includes('_DEFENCE')) {
          continue;
        }
        for (const { key, value } of this.constants) {
          contentLinesPre[i] = contentLinesPre[i].replaceAll(key, value);
        }
      }
      content = contentLinesPre.join('\n');
    }
    try {
      content = Minify(content, {
        RenameVariables: true,
        RenameGlobals: false,
        SolveMath: true,
        Indentation: ' ',
      });
      // there are lines generated by the minifier, so remove them
      let contentLines = content
        .split('\n')
        .map((line) => line.trim())
        .filter((line) => line.length > 0);
      const commentStartLine = contentLines.findIndex(
        (line) => line === '--[[',
      );
      if (commentStartLine >= 0) {
        const commentEndLine = contentLines.findIndex(
          (line, index) => index > commentStartLine && line === '--]]',
        );
        if (commentEndLine > commentStartLine) {
          contentLines = contentLines.filter(
            (_, index) => index < commentStartLine || index > commentEndLine,
          );
        } else {
          contentLines = contentLines.filter(
            (_, index) => index < commentStartLine,
          );
        }
      }
      content = contentLines
        .filter((line) => !line.startsWith('--'))
        .join('\n')
        .replace(/[(,](\d+[+\-*/&|^~%])+\d+[),]/g, (expr) => {
          const firstChar = expr[0];
          const lastChar = expr[expr.length - 1];
          const innerExpr = expr.slice(1, -1);
          const num = eval(innerExpr) as number;
          const dec = num.toString(10);
          const hex = '0x' + num.toString(16).toLowerCase();
          return `${firstChar}${hex.length < dec.length ? hex : dec}${lastChar}`;
        });
      await fs.promises.writeFile(absPath, content, 'utf-8');
    } catch (e) {
      console.error(`Failed to minify ${filePath}:`, e);
      return false;
    }
    return true;
  }

  async run() {
    await this.loadConstants();
    // find all lua files recursively
    const luaFiles: string[] = [];
    const walk = async (dir: string) => {
      const files = await fs.promises.readdir(dir);
      for (const file of files) {
        const fullPath = path.join(dir, file);
        const stat = await fs.promises.stat(fullPath);
        if (stat.isDirectory()) {
          await walk(fullPath);
        } else if (stat.isFile() && file.endsWith('.lua')) {
          luaFiles.push(path.relative(this.options.root, fullPath));
        }
      }
    };
    await walk(this.options.root);
    console.log(`Found ${luaFiles.length} lua files`);
    const fails: string[] = [];
    for (const file of luaFiles) {
      console.log(`Minifying ${file}...`);
      if (!(await this.minifyFile(file))) {
        fails.push(file);
      }
    }
    console.log(
      'Done. Success:',
      luaFiles.length - fails.length,
      'Fail:',
      fails.length,
    );
    if (fails.length > 0) {
      console.log('Failed files:');
      for (const file of fails) {
        console.log('  ', file);
      }
    }
  }
}
