import { ConsoleLogger, Injectable } from '@nestjs/common';
import { HttpService } from '@nestjs/axios';
import { TwintData, TwintReturn } from '../dto/hami.dto';
import { ConfigService } from '@nestjs/config';
import { lastValueFrom } from 'rxjs';
import { AxiosRequestConfig } from 'axios';
import _ from 'lodash';

@Injectable()
export class HamiFetcherService extends ConsoleLogger {
  private readonly twintUrl: string = this.config.get('twintUrl');
  private readonly twintToken: string = this.config.get('twintToken');
  private readonly hamiUsername: string = this.config.get('hamiUsername');
  private readonly hamiBlacklistSearch: string = this.config.get(
    'hamiBlacklistSearch',
  );
  constructor(private http: HttpService, private config: ConfigService) {
    super('HamiFetcherService');
  }

  axiosConfig(): AxiosRequestConfig {
    return {
      headers: {
        Authorization: `Bearer ${this.twintToken}`,
      },
      timeout: 600000,
      responseType: 'json',
    };
  }

  async fetchBlacklistSingle(Since?: string): Promise<TwintData[]> {
    try {
      const { data } = await lastValueFrom(
        this.http.post<TwintReturn>(
          `${this.twintUrl}/Profile`,
          {
            Username: this.hamiUsername,
            Since,
          },
          this.axiosConfig(),
        ),
      );
      return data.data;
    } catch (e) {
      this.error(`Failed to fetch blacklist since ${Since}: ${e.message}`);
      return [];
    }
  }

  async fetchBlacklist(Since?: string): Promise<TwintData[]> {
    const result = await this.fetchBlacklistSingle(Since);
    return _.uniqBy(
      result.filter((t) => t.tweet?.includes(this.hamiBlacklistSearch)),
      (tweet) => tweet.id_str,
    );
  }
}
