import { ConsoleLogger, Injectable, OnModuleInit } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { HttpService } from '@nestjs/axios';
import { BlacklistService } from '../blacklist/blacklist.service';
import { lastValueFrom } from 'rxjs';
import { HamiData } from '../entities/blacklist.entity';
import { scheduleJob } from 'node-schedule';

@Injectable()
export class FetchService extends ConsoleLogger implements OnModuleInit {
  private readonly fetcherUrl =
    this.config.get<string>('FETCHER_URL') || 'http://wenyuanwall-fetcher:3000';
  private readonly cron = this.config.get<string>('CRON');
  constructor(
    private http: HttpService,
    private config: ConfigService,
    private blacklist: BlacklistService,
  ) {
    super('FetchService');
  }

  onModuleInit() {
    this.prepare().then();
  }

  async prepare() {
    if (!this.cron) {
      return;
    }
    await this.fetch();
    scheduleJob(this.cron, () => this.fetch());
  }

  async fetch() {
    const since = await this.blacklist.getSinceTime();
    this.log(`Fetching entries since ${since}...`);
    try {
      const {
        data: { data },
      } = await lastValueFrom(
        this.http.get<{ data: HamiData[] }>(
          `${this.fetcherUrl}/api/blacklist`,
          { responseType: 'json', timeout: 7200000, params: { since } },
        ),
      );
      this.log(`Fetched ${data.length} entries.`);
      await this.blacklist.saveEntries(data);
      this.log(`Done.`);
    } catch (e) {
      this.error(`Failed to fetch entries: ${e.message}`);
    }
  }
}
