import {
  BoolColumn,
  IdBase,
  IntColumn,
  NotColumn,
  NotQueryable,
  QueryColumn,
  QueryCondition,
  QueryEqual,
  RestfulFactory,
  StringColumn,
} from 'nicot';
import { Entity, Index, ManyToOne, OneToMany } from 'typeorm';

@Entity()
export class WhitelistAccount extends IdBase() {
  @NotQueryable()
  @StringColumn(255, {
    required: true,
    description: 'name',
  })
  name: string;

  @Index()
  @QueryEqual()
  @StringColumn(11, {
    required: true,
    description: 'QQ account number.',
  })
  account: string;

  @NotQueryable()
  @StringColumn(255, {
    description: 'Twitter ID',
  })
  twitter: string;

  @NotQueryable()
  @StringColumn(255, {
    description: 'Fee',
  })
  fee: string;

  @NotQueryable()
  @StringColumn(255, {
    description: 'type',
  })
  type: string;

  @NotQueryable()
  @StringColumn(255, {
    description: 'forbidden',
  })
  forbidden: string;

  @BoolColumn({
    description: 'false => not available in random',
    default: true,
  })
  @NotQueryable()
  enabled: boolean;

  @QueryColumn()
  @QueryCondition((obj, qb, entityName, key) => {
    if (obj[key] === 1) {
      qb.andWhere(`${entityName}.enabled = TRUE`)
        .andWhere(`${entityName}.refId IS NULL`)
        .andWhere(`${entityName}.account IS NOT NULL`)
        .andWhere(`LENGTH(${entityName}.account) > 0`)
        .addSelect('RANDOM()', 'rand')
        .orderBy('rand')
    }
  })
  random: number;

  getRecordsPerPage() {
    return this.recordsPerPage || 5;
  }

  @NotQueryable()
  @IntColumn('bigint', {
    unsigned: true,
    description: 'Reference ID',
  })
  refId: number;

  @NotColumn({
    description: 'Referenced account',
  })
  @ManyToOne(() => WhitelistAccount, e => e.refs)
  ref: WhitelistAccount;

  @NotColumn({
    description: 'Referenced by these accounts',
  })
  @OneToMany(() => WhitelistAccount, e => e.ref)
  refs: WhitelistAccount[];
}

export const WhitelistAccountFactory = new RestfulFactory(WhitelistAccount, {
  relations: [
    'ref',
    'ref.refs',
    'refs',
  ],
});
